%% Selection of Input File

tic
clear all
clc

fprintf('LIfe CYcle cost Analyzer - Alpha 2\n')
fprintf('-----------------------------------------\n')
fprintf('-----------------------------------------\n')
fprintf('Please select input file...\n')
fprintf('\n')

addpath('C:\Program Files\MATLAB\TIXI 3.3.0\share\tixi3\matlab');

[filename, path] = uigetfile({'*.xlsx'});     % Excel I/O file selection (Ex. LCC MODEL.xlsx)
file = fullfile(path,filename);               % Building full xlsx file path

%%%%%%%%%%%%%%%%%%%%%%%
%% RDTE & PROD COSTS %%
%%%%%%%%%%%%%%%%%%%%%%%

%% PRE-PROCESSING

%% Weight Input
% In the xlsx file user must select mass imput source: 'Mass Breakdown' worksheet
% of CPACS file. Reading functions will be used accordingly

[~,Mass_Input] = xlsread(file, 'Mass Breakdown', 'I3');     % Reading user's input choice
Mass_Input = string(Mass_Input);

if isequal(Mass_Input ,'CPACS')     % Start reading from cpacs file if user selected it as input mode. Data not included in cpasc will be read from 'Mass Breakdown' worksheet

    fprintf('Please select CPACS file...\n\n')

    [filename_xml, path_xml] = uigetfile({'*.xml'});    % .xml cpacs file selection
    file_xml = fullfile(path_xml,filename_xml);     % Building full xml file path
    handle = tixiOpenDocument(file_xml);            % Cpacs file opening

    [~,Perc_Input] = xlsread(file, 'Mass Percentages', 'I3');     % Reading user's input choice for mass percentages
    Perc_Input = string(Perc_Input);
    if isequal(Perc_Input,'Reference')
        perc = xlsread(file,'Percentual','E4:E70');     % Reading 'Mass Percentages' worksheet for Cpacs input, reference % based on ATR 42
    else
        perc = xlsread(file,'Percentual','F4:F70');     % Reading 'Mass Percentages' worksheet for Cpacs input, custom % chosen by the user
    end

    Quantity = zeros(67,1);     % Initializing Quantity vector

    fprintf('Importing masses...\n')
    
    try     % WING, reading mass from cpacs file
        Weight(1,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mStructure/mWingsStructure/mWingStructure[1]/massDescription/mass');
    catch   % If component isn't included in cpacs file, read it from xlsx input file
        Weight(1,1) = xlsread(file,'Mass Breakdown','E4');
    end
    
    try     % FUSELAGE
        Weight(2,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mStructure/mFuselagesStructure/mFuselageStructure/massDescription/mass');
    catch   
        Weight(2,1) = xlsread(file,'Mass Breakdown','E5');
    end
    
    try     % HORIZONTAL TAIL
        Weight(3,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mStructure/mWingsStructure/mWingStructure[2]/massDescription/mass');
    catch   
        Weight(3,1) = xlsread(file,'Mass Breakdown','E6');
    end
    
    try     % VERTICAL TAIL
        Weight(4,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mStructure/mWingsStructure/mWingStructure[3]/massDescription/mass');
    catch   
        Weight(4,1) = xlsread(file,'Mass Breakdown','E7');
    end
    
    Weight(5,1) = xlsread(file,'Mass Breakdown','E8');  % NACELLES (not included in CPACS)
    Quantity(5,1) = xlsread(file,'Mass Breakdown','F8');    % Number of elements of the same weight (If total weight in entered, quantity is equal to 1)

    try     % MAIN LG
        Weight(6,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mStructure/mLandingGears/mMainGears/massDescription/mass');
    catch   
        Weight(6,1) = xlsread(file,'Mass Breakdown','E9');
    end
    
    try     % NOSE LG
        Weight(7,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mStructure/mLandingGears/mNoseGears/massDescription/mass');
    catch   
        Weight(7,1) = xlsread(file,'Mass Breakdown','E10');
    end
    
    try     % TRADITIONAL POWERPLANT (Equipped Engine; Engine Control)
        Weight(8:9,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mPowerUnits/massDescription/mass'.*perc(8:9));
    catch   
        Weight(8:9,1) = xlsread(file,'Mass Breakdown','E11:E12');
    end
    
    Weight(10:19,1) = xlsread(file,'Mass Breakdown','E13:E22');  % HYBRID POWERPLANT COMPONENTS
    Quantity(10:19,1) = xlsread(file,'Mass Breakdown','F13:F22');  

    try     % APU
        Weight(20,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mSystems/mAuxillaryPowerUnit/massDescription/mass');
    catch   
        Weight(20,1) = xlsread(file,'Mass Breakdown','E23');
    end
    
    try     % HYDRAULIC GENERATION
        Weight(21,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mSystems/mHydraulicGeneration/massDescription/mass');
    catch   
        Weight(21,1) = xlsread(file,'Mass Breakdown','E24');
    end
    
    try     % HYDRAULIC DISTRIBUTION
        Weight(22,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mSystems/mHydraulicDistribution/massDescription/mass');
    catch   
        Weight(22,1) = xlsread(file,'Mass Breakdown','E25');
    end
    
    try     % FUEL SYSTEM (Refuelling; Fuelling)
        Weight(23:24,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mPowerUnits/mFuelSystem/massDescription/mass'.*perc(23:24));
    catch   
        Weight(23:24,1) = xlsread(file,'Mass Breakdown','E26:E27');
    end
    
    try     % ECS
        Weight(25,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mSystems/mAirConditioning/massDescription/mass');
    catch   
        Weight(25,1) = xlsread(file,'Mass Breakdown','E28');
    end
    
    try     % DE-ICING (Wing; Tail)
        Weight(26:27,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mSystems/mDeIcing/massDescription/mass'.*perc(26:27));
    catch   
        Weight(26:27,1) = xlsread(file,'Mass Breakdown','E29:E30');
    end
    
    try     % FIRE PROTECTION
        Weight(28,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mSystems/mFireProtection/massDescription/mass');
    catch   
        Weight(28,1) = xlsread(file,'Mass Breakdown','E31');
    end
    
    try     % ECS
        Weight(29,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mSystems/mFlightControls/massDescription/mass');
    catch   
        Weight(29,1) = xlsread(file,'Mass Breakdown','E32');
    end
    
    Weight(30,1) = xlsread(file,'Mass Breakdown','E33');  % FLIGHT COMPARTMENT FURNISHING
    Quantity(30,1) = xlsread(file,'Mass Breakdown','F33');

    try     % AUTOMATIC FLIGHT SYSTEM
        Weight(31,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mSystems/mAutomaticFlightSystem/massDescription/mass');
    catch   
        Weight(31,1) = xlsread(file,'Mass Breakdown','E34');
    end
    
    try     % NAVIGATION SYSTEM
        Weight(32:41,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mSystems/mNavigation/massDescription/mass'.*perc(32:41));
    catch   
        Weight(32:41,1) = xlsread(file,'Mass Breakdown','E35:E44');
    end
    
    Weight(42:43,1) = xlsread(file,'Mass Breakdown','E45:E46');  % MISSION SW; AIR DATA SW
    Quantity(42:43,1) = xlsread(file,'Mass Breakdown','F45:F46');

    try     % COMUNICATION SYSTEM
        Weight(44:50,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mSystems/mCommunication/massDescription/mass'.*perc(44:50));
    catch   
        Weight(44:50,1) = xlsread(file,'Mass Breakdown','E47:E55');
    end
    
    try     % ELECTRIAL GENERATION
        Weight(51,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mSystems/mElectricalGeneration/massDescription/mass');
    catch   
        Weight(51,1) = xlsread(file,'Mass Breakdown','E54');
    end
    
    try     % ELECTRICAL DISTRIBUTION
        Weight(52,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mSystems/mElectricalDistribution/massDescription/mass');
    catch   
        Weight(52,1) = xlsread(file,'Mass Breakdown','E55');
    end
    
    Weight(53:59,1) = xlsread(file,'Mass Breakdown','E56:E62');  % HYBRID ELECTRICAL COMPONENTS
    Quantity(53:59,1) = xlsread(file,'Mass Breakdown','F56:F62'); 

    try     % FURNISHINGS and EQUIPMENTS
        Weight(60:67,1) = tixiGetDoubleElement(handle,'/cpacs/vehicles/aircraft/model/analyses/massBreakdown/mOEM/mEM/mFurnishing/massDescription/mass'.*perc(60:67));
    catch   
        Weight(60:67,1) = xlsread(file,'Mass Breakdown','E63:E70');
    end

    Elem_cpacs = [1:4, 6:9, 20:29, 31:41, 44:52, 60:67];    % Setting quantity for elements taken from Cpacs
    for i = 1:46
        if Weight(Elem_cpacs(i)) > 0
            Quantity(Elem_cpacs(i)) = 1;
        end
    end

else            % Start reading from 'Mass Breakdown' worksheet if user selected it as input mode
    fprintf('Importing masses...\n')
    Weight = xlsread(file,'Mass Breakdown','E4:E70');
    Quantity = xlsread(file,'Mass Breakdown','F4:F70');
end

Quantity(67:94) = 1;    % Setting Quantity for systems

flag_powerplant = 0;    % Initilizing flags for hybrid systems
flag_electrical = 0;

for i=10:19                     % Check if any hybrid powerplant component is present,  
    if Quantity(i)>0            % if so, updated CERs for hybrid systems will be overwritten on traditional ones
        flag_powerplant = 1;
        break
    end
end

for i=53:59                 % Check is any hybrid electrical component is present
    if Quantity(i)>0        % if so, updated CERs for hybrid systems will be overwritten on traditional ones
        flag_electrical = 1;
        break
    end
end

%% General Input 
% User must write or select these values on the first worksheet of the .xlsx file
fprintf('Importing parameters...\n')
n_prod = xlsread(file,'General Input','B8');    % Reading data from 'General Input' worksheet
CEF = xlsread(file,'General Input','B16');
[~,Cur] = xlsread(file,'General Input', 'B17');
Cur = string(Cur);  
Change = xlsread(file,'General Input', 'B18');

InflactionUSD_2023vs2017 = 1.25;    % Factor to update costs from 2017 to 2023 considering inflaction

%% Parameters' Input 
% Parameters affecting cost at component, assembly and total level
% These parameters can be manually written by the user in the .xlsx file
Comp_Parameters = xlsread(file,'Component Parameters','E4:I67');        % Reading input data from 'Component Parameters' worksheet
Ass_Parameters = xlsread(file,'Assembly Parameters','C4:D29');          % Reading input data from 'Assembly Parameters' worksheet
General_Parameters = xlsread(file,'General Input','B10:B14');           % Reading input data from 'General Parameters' worksheet
Software_Parameters = xlsread(file,'Component Parameters','E70:H71');   % Reading input data from 'Component Parameters' worksheet for software elements

%% LC input
% Learning Curve gradient for each component and system. It affects PROD costs
LC_comp = zeros(67,1);      % Initilizing LC array
LC_comp(1:41,1) = xlsread(file,'LC','E4:E44');      % LC for components
LC_comp(42:43,1) = 1;                               % LC for software is set to 1
LC_comp(44:67,1) = xlsread(file,'LC','E45:E68');    % LC for components
LC_ass = xlsread(file,'LC','E71:E96');              % LC for assemblies

%% RDTE - PROD CERs Loading
% Coefficient for all the CERs are stored in CERs.xlsx. This file must be saved in the same folder as LCC_MODEL.m.
fprintf('Importing CERs coefficients...\n')
    
%%% Coefficients for Components' Base Cost CERs based on Weight (Poly1 type)
RDTE_BaseCost_comp_coeff = xlsread('CERs.xlsx', 'CERs RDTE', 'D3:E69');
PROD_BaseCost_comp_coeff = xlsread('CERs.xlsx', 'CERs PROD', 'D3:E69');

%%% Coefficients for Assemblies' Base Cost CERs based on Weight (Poly1 type)
RDTE_BaseCost_ass_coeff = xlsread('CERs.xlsx', 'CERs RDTE', 'D70:E96');
PROD_BaseCost_ass_coeff = xlsread('CERs.xlsx', 'CERs PROD', 'D70:E96');

%%% Coefficients for Hybrid Configurations Assemblies' Base Cost CERs based on Weight (Poly1 type)
RDTE_BaseCost_Hyb_ass_coeff = xlsread('CERs.xlsx', 'CERs RDTE', 'D98:E102');
PROD_BaseCost_Hyb_ass_coeff = xlsread('CERs.xlsx', 'CERs PROD', 'D98:E102');

%%% Coefficients for Components' Base Cost CERs based on Weight (Poly2 type)
RDTE_BaseCost_comp_coeff_P2 = xlsread('CERs.xlsx', 'CERs RDTE', 'F3:H69');
PROD_BaseCost_comp_coeff_P2 = xlsread('CERs.xlsx', 'CERs PROD', 'F3:H69');

%%% Coefficients for Assemblies' Base Cost CERs based on Weight (Poly2 type)
RDTE_BaseCost_ass_coeff_P2 = xlsread('CERs.xlsx', 'CERs RDTE', 'F70:H96');
PROD_BaseCost_ass_coeff_P2 = xlsread('CERs.xlsx', 'CERs PROD', 'F70:H96');

%%% Coefficients for Hybrid Configurations Assemblies' Base Cost CERs based on Weight (Poly2 type)
RDTE_BaseCost_Hyb_ass_coeff_P2 = xlsread('CERs.xlsx', 'CERs RDTE', 'F98:H102');
PROD_BaseCost_Hyb_ass_coeff_P2 = xlsread('CERs.xlsx', 'CERs PROD', 'F98:H102');

%%% X0: intersection (Poly2 -> Poly1) or Vertex (Poly2)
X0_comp_RDTE = xlsread('CERs.xlsx', 'CERs RDTE', 'I3:I69');
X0_comp_PROD = xlsread('CERs.xlsx', 'CERs PROD', 'I3:I69');
X0_ass_RDTE = xlsread('CERs.xlsx', 'CERs RDTE', 'I70:I96');
X0_ass_PROD = xlsread('CERs.xlsx', 'CERs PROD', 'I70:I96');
X0_Hyb_ass_RDTE = xlsread('CERs.xlsx', 'CERs RDTE', 'I98:I102');
X0_Hyb_ass_PROD = xlsread('CERs.xlsx', 'CERs PROD', 'I98:I102');

%%% Curve Type
% Poly1: straight line from linear regression 
% Poly2: parabolic curve till vertex (X0), than constant
% P2-P1: parabola till intersection (X0), than straight line
% P1-P2: straight line till intersection (X0), than parabola
[~,RDTE_Type_comp] = xlsread('CERs.xlsx', 'CERs RDTE', 'J3:J69');
[~,RDTE_Type_ass] = xlsread('CERs.xlsx', 'CERs RDTE', 'J70:J96');
[~,RDTE_Type_Hyb_ass] = xlsread('CERs.xlsx', 'CERs RDTE', 'J98:J102');
RDTE_Type_comp = string(RDTE_Type_comp);
RDTE_Type_ass = string(RDTE_Type_ass);
RDTE_Type_Hyb_ass = string(RDTE_Type_Hyb_ass);

[~,PROD_Type_comp] = xlsread('CERs.xlsx', 'CERs PROD', 'J3:J69');
[~,PROD_Type_ass] = xlsread('CERs.xlsx', 'CERs PROD', 'J70:J96');
[~,PROD_Type_Hyb_ass] = xlsread('CERs.xlsx', 'CERs PROD', 'J98:J102');
PROD_Type_comp = string(PROD_Type_comp);
PROD_Type_ass = string(PROD_Type_ass);
PROD_Type_Hyb_ass = string(PROD_Type_Hyb_ass);

%%% Coefficients for Components' Parameters CERs
% RDTE(:, 1:2) - Engineering complexity for Components (Columns K-L)
% RDTE(:, 3:5) - % New Design (M-O)
% RDTE(:, 6:8) - % Design Repeat (P-R)
% RDTE(:, 9:12) - Manufacturing Complexity (S-V)
% They don't affect Fuel Cell (Row 20 on the xlsx) and Softwares (44-45)
RDTE_Comp_Param(1:17,:) = xlsread('CERs.xlsx', 'CERs RDTE', 'K3:V19');
RDTE_Comp_Param(18:40,:) = xlsread('CERs.xlsx', 'CERs RDTE', 'K21:V43');
RDTE_Comp_Param(41:64,:) = xlsread('CERs.xlsx', 'CERs RDTE', 'K46:V69');
% PROD(:, 1:4) - Manufacturing Complexity (S-V)
PROD_Comp_Param(1:17,:) = xlsread('CERs.xlsx', 'CERs PROD', 'S3:V19');
PROD_Comp_Param(18:40,:) = xlsread('CERs.xlsx', 'CERs PROD', 'S21:V43');
PROD_Comp_Param(41:64,:) = xlsread('CERs.xlsx', 'CERs PROD', 'S46:V69');

%%% Coefficients for Software CERs
RDTE_SW_DesRep = xlsread('CERs.xlsx', 'CERs SW', 'D2:G2');      % Design Repeat(%)
RDTE_SW_DevTeam = xlsread('CERs.xlsx', 'CERs SW', 'D3:E3');     % Development Team Complexity
RDTE_SW_FunComp = xlsread('CERs.xlsx', 'CERs SW', 'D4:F4');     % Functional Complexity
PROD_SW_FunComp = xlsread('CERs.xlsx', 'CERs SW', 'D5:F5');     % Functional Complexity

%%% Coefficients for Assemblies' Parameters CERs
% (:, 1:3) - System complexity (W-Y)
% (:, 4:6) - Engineering Complexity for Systems (Z-AB)
RDTE_Ass_Param = xlsread('CERs.xlsx', 'CERs RDTE', 'W70:AB95');
RDTE_Ass_Hyb_Param = xlsread('CERs.xlsx', 'CERs RDTE', 'W98:AB101');

%%% Coefficients for General Parameters CERs
% Multiple Site Development
RDTE_Comp_Site = xlsread('CERs.xlsx', 'CERs RDTE', 'AC44:AE45');    % Software
RDTE_Ass_Site = xlsread('CERs.xlsx', 'CERs RDTE', 'AC70:AE93');     % Assemblies
RDTE_Hyb_Ass_Site = xlsread('CERs.xlsx', 'CERs RDTE', 'AC99:AE99'); % Electrical System for Hybrid Config. 

% Management Team Complexity
RDTE_Prog_Team = xlsread('CERs.xlsx', 'CERs RDTE', 'AF96:AH96');         % Total Program RDTE
RDTE_Hyb_Prog_Team = xlsread('CERs.xlsx', 'CERs RDTE', 'AF102:AH102');   % Total Program for Hybrid Config. RDTE
PROD_Prog_Team = xlsread('CERs.xlsx', 'CERs PROD', 'AF96:AH96');         % Total Program PROD
PROD_Hyb_Prog_Team = xlsread('CERs.xlsx', 'CERs PROD', 'AF102:AH102');   % Total Program for Hybrid Config. PROD

% Management Organizational Productivity
RDTE_Prog_Org = xlsread('CERs.xlsx', 'CERs RDTE', 'AI96:AK96');         % Total Program
RDTE_Hyb_Prog_Org = xlsread('CERs.xlsx', 'CERs RDTE', 'AI102:AK102');   % Total Program for Hybrid Config.

% Project Complexity Factor
RDTE_Prog_ProjComp = xlsread('CERs.xlsx', 'CERs RDTE', 'AL96:AM96');        % Total Program RDTE
RDTE_Hyb_Prog_ProjComp = xlsread('CERs.xlsx', 'CERs RDTE', 'AL102:AM102');  % Total Program for Hybrid Config. RDTE
PROD_Prog_ProjComp = xlsread('CERs.xlsx', 'CERs PROD', 'AL96:AM96');        % Total Program PROD
PROD_Hyb_Prog_ProjComp = xlsread('CERs.xlsx', 'CERs PROD', 'AL102:AM102');  % Total Program for Hybrid Config. PROD

% Number of Prototypes
% It doesn't affect Fuel Cell (Row 20 on the xlsx) and Softwares (44-45)
RDTE_Comp_Proto(1:17,:) = xlsread('CERs.xlsx', 'CERs RDTE', 'AN3:AO19');
RDTE_Comp_Proto(18:40,:) = xlsread('CERs.xlsx', 'CERs RDTE', 'AN21:AO43');
RDTE_Comp_Proto(41:64,:) = xlsread('CERs.xlsx', 'CERs RDTE', 'AN46:AO69');
PROD_Comp_Proto(1:17,:) = xlsread('CERs.xlsx', 'CERs PROD', 'AN3:AP19');
PROD_Comp_Proto(18:40,:) = xlsread('CERs.xlsx', 'CERs PROD', 'AN21:AP43');
PROD_Comp_Proto(41:64,:) = xlsread('CERs.xlsx', 'CERs PROD', 'AN46:AP69');

%%% CERs update for hybrid configurations
% If hybrid components are planned, related Assemblies, Total Aircraft and
% Total Program CERs' coefficient must be owerwritten on traditional ones

if flag_powerplant==1       % If hybrid powerplant components are present
    RDTE_BaseCost_ass_coeff(3,:) = RDTE_BaseCost_Hyb_ass_coeff(1,:);        % Powerplant System RDTE Base Cost (Poly1)
    RDTE_BaseCost_ass_coeff_P2(3,:) = RDTE_BaseCost_Hyb_ass_coeff_P2(1,:);  % Powerplant System RDTE Base Cost (Poly2)
    X0_ass_RDTE(3,:) = X0_Hyb_ass_RDTE(1,:);                                % Powerplant System RDTE Curve Intersection
    RDTE_Type_ass(3) = RDTE_Type_Hyb_ass(1);                                % Powerplant System RDTE Curve Type     
    PROD_BaseCost_ass_coeff(3,:) = PROD_BaseCost_Hyb_ass_coeff(1,:);        % Powerplant System PROD Base Cost (Poly1)
    PROD_BaseCost_ass_coeff_P2(3,:) = PROD_BaseCost_Hyb_ass_coeff_P2(1,:);  % Powerplant System PROD Base Cost (Poly2)
    X0_ass_PROD(3,:) = X0_Hyb_ass_PROD(1,:);                                % Powerplant System PROD Curve Intersection
    PROD_Type_ass(3) = PROD_Type_Hyb_ass(1);                                % Powerplant System PROD Curve Type
    RDTE_Ass_Param(3,:) = RDTE_Ass_Hyb_Param(1,:);          % Powerplant Assembly's Parameters CERs (System + Engineering Complexity)
end

if flag_electrical==1       % If hybrid electrical components are present
    RDTE_BaseCost_ass_coeff(15:16,:) = RDTE_BaseCost_Hyb_ass_coeff(2:3,:);          % Electrical System and Total Subsystems RDTE Base Cost (Poly1)
    RDTE_BaseCost_ass_coeff_P2(15:16,:) = RDTE_BaseCost_Hyb_ass_coeff_P2(2:3,:);    % Electrical System and Total Subsystems RDTE Base Cost (Poly2)
    X0_ass_RDTE(15:16,:) = X0_Hyb_ass_RDTE(2:3,:);                                  % Electrical System and Total Subsystems RDTE Curve Intersection
    RDTE_Type_ass(15:16) = RDTE_Type_Hyb_ass(2:3);                                  % Electrical System and Total Subsystems RDTE Curve Type   
    PROD_BaseCost_ass_coeff(15:16,:) = PROD_BaseCost_Hyb_ass_coeff(2:3,:);          % Electrical System and Total Subsystems PROD Base Cost (Poly1)
    PROD_BaseCost_ass_coeff_P2(15:16,:) = PROD_BaseCost_Hyb_ass_coeff_P2(2:3,:);    % Electrical System and Total Subsystems PROD Base Cost (Poly2)
    X0_ass_PROD(15:16,:) = X0_Hyb_ass_PROD(2:3,:);                                  % Electrical System and Total Subsystems PROD Curve Intersection
    PROD_Type_ass(15:16) = PROD_Type_Hyb_ass(2:3);                                  % Electrical System and Total Subsystems PROD Curve Type
    RDTE_Ass_Param(15:16,:) = RDTE_Ass_Hyb_Param(2:3,:);    % Electrical System and Total Subsystems Assemblies' Parameters CERs (System + Engineering Complexity)
    RDTE_Ass_Site(15,:) = RDTE_Hyb_Ass_Site;                % Electrical System Multiple Site Development CER's Coefficients
end

if flag_powerplant==1 || flag_electrical==1     % If either hybrid powerplant or electrical components are present
    RDTE_BaseCost_ass_coeff(26:27,:) = RDTE_BaseCost_Hyb_ass_coeff(4:5,:);          % Total Aircraft and Program RDTE Base Cost (Poly1)
    RDTE_BaseCost_ass_coeff_P2(26:27,:) = RDTE_BaseCost_Hyb_ass_coeff_P2(4:5,:);    % Total Aircraft and Program RDTE Base Cost (Poly2)
    X0_ass_RDTE(26:27,:) = X0_Hyb_ass_RDTE(4:5,:);                                  % Total Aircraft and Program RDTE Curve Intersection
    RDTE_Type_ass(26:27) = RDTE_Type_Hyb_ass(4:5);                                  % Total Aircraft and Program RDTE Curve Type  
    PROD_BaseCost_ass_coeff(26:27,:) = PROD_BaseCost_Hyb_ass_coeff(4:5,:);          % Total Aircraft and Program PROD Base Cost (Poly1)
    PROD_BaseCost_ass_coeff_P2(26:27,:) = PROD_BaseCost_Hyb_ass_coeff_P2(4:5,:);    % Total Aircraft and Program PROD Base Cost (Poly2)
    X0_ass_PROD(26:27,:) = X0_Hyb_ass_PROD(4:5,:);                                  % Total Aircraft and Program PROD Curve Intersection
    PROD_Type_ass(26:27) = PROD_Type_Hyb_ass(4:5);                                  % Total Aircraft and Program PROD Curve Type
    
    RDTE_Ass_Param(26,:) = RDTE_Ass_Hyb_Param(4,:);     % Total Aircraft Assembly's Parameters CERs (System + Engineering Complexity)
    RDTE_Prog_Team = RDTE_Hyb_Prog_Team;                % Total Program Management Team Complexity RDTE CER's Coefficients
    PROD_Prog_Team = PROD_Hyb_Prog_Team;                % Total Program Management Team Complexity PROD CER's Coefficients
    RDTE_Prog_Org = RDTE_Hyb_Prog_Org;                  % Total Program Management Organizational Productivity CER's Coefficients
    RDTE_Prog_ProjComp = RDTE_Hyb_Prog_ProjComp;        % Total Program Project Complexity Factor RDTE CER's Coefficients
    PROD_Prog_ProjComp = PROD_Hyb_Prog_ProjComp;        % Total Program Project Complexity Factor PROD CER's Coefficients
end

%% PROCESSING

%% RDTE - PROD Components' Base Cost
fprintf('\nProcessing Cost Estimation for Components...\n')

for i=1:length(Weight)      % Cost estimation component by component
    switch RDTE_Type_comp(i)        % RDTE Cost estimation using different CER for each component, based on Curve Type
        case 'Poly1'    % Straight Line for any weight
            RDTE_Comp_BaseCost(i,1) = Weight(i)*Quantity(i)*RDTE_BaseCost_comp_coeff(i,1) + RDTE_BaseCost_comp_coeff(i,2);
        case 'Poly2'    % Parabola till vertex, than constant
            if Weight(i) < X0_comp_RDTE(i)      % If component's weight is less than vertex -> cost calculation with quadratic polynomial
                RDTE_Comp_BaseCost(i,1) =  RDTE_BaseCost_comp_coeff_P2(i,1)*(Weight(i)*Quantity(i))^2 + RDTE_BaseCost_comp_coeff_P2(i,2)*Weight(i)*Quantity(i) + RDTE_BaseCost_comp_coeff_P2(i,3);
            else    % If component's weight is beyond vertex -> constant cost regardless of weight (cost of vertex value)
                RDTE_Comp_BaseCost(i,1) =  RDTE_BaseCost_comp_coeff_P2(i,1)*(X0_comp_RDTE(i))^2 + RDTE_BaseCost_comp_coeff_P2(i,2)*X0_comp_RDTE(i) + RDTE_BaseCost_comp_coeff_P2(i,3);
            end
        case 'P2-P1'    % Parabola till intersection, than straight line
            if Weight(i) < X0_comp_RDTE(i)  % If component's weight is less than intersection -> cost calculation with quadratic polynomial
                RDTE_Comp_BaseCost(i,1) =  RDTE_BaseCost_comp_coeff_P2(i,1)*(Weight(i)*Quantity(i))^2 + RDTE_BaseCost_comp_coeff_P2(i,2)*Weight(i)*Quantity(i) + RDTE_BaseCost_comp_coeff_P2(i,3);
            else        % If component's weight is beyond vertex -> straight line
                RDTE_Comp_BaseCost(i,1) = Weight(i)*Quantity(i)*RDTE_BaseCost_comp_coeff(i,1) + RDTE_BaseCost_comp_coeff(i,2);
            end
    end
    switch PROD_Type_comp(i)     % PROD Cost estimation, follows the same process as RDTE costs with specific CERs
        case 'Poly1'
            PROD_Comp_BaseCost(i,1) = Weight(i)*Quantity(i)*PROD_BaseCost_comp_coeff(i,1) + PROD_BaseCost_comp_coeff(i,2);
        case 'Poly2'
            if Weight(i) < X0_comp_PROD(i)
                PROD_Comp_BaseCost(i,1) =  PROD_BaseCost_comp_coeff_P2(i,1)*(Weight(i)*Quantity(i))^2 + PROD_BaseCost_comp_coeff_P2(i,2)*Weight(i)*Quantity(i) + PROD_BaseCost_comp_coeff_P2(i,3);
            else   
                PROD_Comp_BaseCost(i,1) =  PROD_BaseCost_comp_coeff_P2(i,1)*(X0_comp_PROD(i))^2 + PROD_BaseCost_comp_coeff_P2(i,2)*X0_comp_PROD(i) + PROD_BaseCost_comp_coeff_P2(i,3);
            end
        case 'P2-P1'
            if Weight(i) < X0_comp_PROD(i)
                PROD_Comp_BaseCost(i,1) =  PROD_BaseCost_comp_coeff_P2(i,1)*(Weight(i)*Quantity(i))^2 + PROD_BaseCost_comp_coeff_P2(i,2)*Weight(i)*Quantity(i) + PROD_BaseCost_comp_coeff_P2(i,3);
            else
                PROD_Comp_BaseCost(i,1) = Weight(i)*Quantity(i)*PROD_BaseCost_comp_coeff(i,1) + PROD_BaseCost_comp_coeff(i,2);
            end
    end
end

%% RDTE - PROD Parameters Processing for Components
% For each component, the multiplicative factor associated with each parameter that affects its cost is calculated
F_EngComp_RDTE = RDTE_Comp_Param(:,1).*Comp_Parameters(:,1) + RDTE_Comp_Param(:,2);   % Engineering Complexity: F = a*EC + b
F_PercNew_RDTE = RDTE_Comp_Param(:,3).*(Comp_Parameters(:,2).^2) + RDTE_Comp_Param(:,4).*Comp_Parameters(:,2) + RDTE_Comp_Param(:,5);   % Percent. New Design: F = a*%ND^2 + b*%ND + c
F_PercRepeat_RDTE = RDTE_Comp_Param(:,6).*(Comp_Parameters(:,3).^2) + RDTE_Comp_Param(:,7).*Comp_Parameters(:,3) + RDTE_Comp_Param(:,8);    % Percent. Design Repeat: F = a*%DR^2 + b*%DR + c
F_ManufComp_RDTE = RDTE_Comp_Param(:,9).*exp(RDTE_Comp_Param(:,10).*Comp_Parameters(:,4)) + RDTE_Comp_Param(:,11).*exp(RDTE_Comp_Param(:,12).*Comp_Parameters(:,4));    % Manufacturing Complexity RDTE: F = a*exp(b*MC) + c*exp(d*MC)
F_Proto_RDTE = RDTE_Comp_Proto(:,1).*General_Parameters(5) + RDTE_Comp_Proto(:,2);   % Number of Prototypes RDTE: F = a*#P + b

F_ManufComp_PROD = PROD_Comp_Param(:,1).*exp(PROD_Comp_Param(:,2).*Comp_Parameters(:,4)) + PROD_Comp_Param(:,3).*exp(PROD_Comp_Param(:,4).*Comp_Parameters(:,4));    % Manufacturing Complexity PROD: F = a*exp(b*MC) + c*exp(d*MC)
F_Proto_PROD = PROD_Comp_Proto(:,1).*(General_Parameters(5))^2 + PROD_Comp_Proto(:,2)*General_Parameters(5) + PROD_Comp_Proto(:,3);     % Number of Prototypes PROD: F = a*#P^2 + b*#P + c

%%% Softwares have specific parameters
F_SW_Site_RDTE = RDTE_Comp_Site(:,1).*(General_Parameters(1))^2 + RDTE_Comp_Site(:,2).*General_Parameters(1) + RDTE_Comp_Site(:,3);     % Multiple Site Development: F = a*MSD^2 + b*MSD + c
F_SW_DesRep_RDTE = RDTE_SW_DesRep(1).*exp(RDTE_SW_DesRep(2).*Software_Parameters(:,1)) + RDTE_SW_DesRep(3)*exp(RDTE_SW_DesRep(4)*Software_Parameters(:,1));     % Software Design Repeat: F = a*exp(b*%DS) + c*exp(d*%DR)
F_SW_DevTeam_RDTE = RDTE_SW_DevTeam(1).*exp(RDTE_SW_DevTeam(2).*Software_Parameters(:,2));    % Development Team Complexity: F = a*exp(b*DTC)
F_SW_FuncComp_RDTE = RDTE_SW_FunComp(1).*(Software_Parameters(:,3)).^2 + RDTE_SW_FunComp(2).*Software_Parameters(:,3) + RDTE_SW_FunComp(3);     % Functional Complexity RDTE: F = a*FC^2 + b*FC + c
F_SW_FuncComp_PROD = PROD_SW_FunComp(1).*(Software_Parameters(:,3)).^2 + PROD_SW_FunComp(2).*Software_Parameters(:,3) + RDTE_SW_FunComp(3);     % Functional Complexity PROD: F = a*FC^2 + b*FC + c

%% Components' Cost Output
% The estimated Base Cost for each component is multiplied by the factors derived from the parameters
RDTE_COMP_COST(1:17,1) = RDTE_Comp_BaseCost(1:17).*F_EngComp_RDTE(1:17).*F_PercNew_RDTE(1:17).*F_PercRepeat_RDTE(1:17).*F_ManufComp_RDTE(1:17).*F_Proto_RDTE(1:17).*Comp_Parameters(1:17,5);
RDTE_COMP_COST(19:41,1) = RDTE_Comp_BaseCost(19:41).*F_EngComp_RDTE(18:40).*F_PercNew_RDTE(18:40).*F_PercRepeat_RDTE(18:40).*F_ManufComp_RDTE(18:40).*F_Proto_RDTE(18:40).*Comp_Parameters(18:40,5);
RDTE_COMP_COST(42:43) = RDTE_Comp_BaseCost(42:43).*F_SW_Site_RDTE(:).*F_SW_DesRep_RDTE(:).*F_SW_DevTeam_RDTE(:).*F_SW_FuncComp_RDTE(:).*Software_Parameters(:,4);                                       % Software
RDTE_COMP_COST(44:67,1) = RDTE_Comp_BaseCost(44:67).*F_EngComp_RDTE(41:64).*F_PercNew_RDTE(41:64).*F_PercRepeat_RDTE(41:64).*F_ManufComp_RDTE(41:64).*F_Proto_RDTE(41:64).*Comp_Parameters(41:64,5);

PROD_COMP_COST(1:17,1) = PROD_Comp_BaseCost(1:17).*F_ManufComp_PROD(1:17).*F_Proto_PROD(1:17);
PROD_COMP_COST(18,1) = PROD_Comp_BaseCost(18);      % Fuel Cell
PROD_COMP_COST(19:41,1) = PROD_Comp_BaseCost(19:41).*F_ManufComp_PROD(18:40).*F_Proto_PROD(18:40);
PROD_COMP_COST(42:43) = PROD_Comp_BaseCost(42:43).*F_SW_FuncComp_PROD(:);                           % Software
PROD_COMP_COST(44:67,1) = PROD_Comp_BaseCost(44:67).*F_ManufComp_PROD(41:64).*F_Proto_PROD(41:64);

% RDTE cost for Fuel Cell in relation to PROD Cost. Same proportion as the battery but increased by 20%
RDTE_COMP_COST(18,1) = RDTE_COMP_COST(16,1)/PROD_COMP_COST(16,1)*PROD_COMP_COST(18,1)*1.2;      % Fuel Cell

%% PROD Components' Learning Curve
% The cost of the first unit is derived from total PROD cost (600units ref)
a_comp = PROD_COMP_COST./146.2415;      % First Unit Cost (Reference LC = 0.83)
a_comp(18) = PROD_COMP_COST(18)/473.6256;   % First Unit Cost for Fuel Cell (Reference LC = 0.97)

% Total PROD cost for 'n' units is calculated iteratively for each component
PROD_COMP_COST_LC = zeros(length(PROD_COMP_COST),1);        % Initializing total PROD cost array
for i=1:length(PROD_COMP_COST)
    if (i==42 || i==43)
        PROD_COMP_COST_LC(i) = PROD_COMP_COST(i);   % Softwares are not affected by Learning Curve
    else
        for n=1:n_prod
            PROD_COMP_COST_LC(i) = PROD_COMP_COST_LC(i) + a_comp(i)*n^(log(LC_comp(i))/log(2));     % The cost for each unit in added to the total till n_prod is reached 
        end
    end   
end

%% RDTE - PROD Level 1 Assemblies' Reference Base Cost
fprintf('Processing Cost Estimation for Assemblies...\n')

% Level 1 Assemblies are those in gray in the PBS
% The reference value for estimating Base Costs for each assembly is the sum of the costs calculated for the components from which it is formed.
RDTE_Ass_Ref = zeros(26,1);     % Initializing RDTE asselmblies' reference values array
RDTE_Ass_Ref(1) = sum(RDTE_COMP_COST(6:7),"omitnan");   % Landing Gear (sum of Main and Nose)
RDTE_Ass_Ref(4) = RDTE_COMP_COST(20);            % APU System
RDTE_Ass_Ref(5) = sum(RDTE_COMP_COST(21:22),"omitnan"); % Hydraulic System (sum of Generation and Distribution)
RDTE_Ass_Ref(6) = sum(RDTE_COMP_COST(23:24),"omitnan"); % Fuel System (sum of Refuelling and Fuelling)
RDTE_Ass_Ref(7) = RDTE_COMP_COST(25);            % ECS
RDTE_Ass_Ref(8) = sum(RDTE_COMP_COST(26:27),"omitnan"); % De-Icing System (sum of Wing and Tail)
RDTE_Ass_Ref(9) = RDTE_COMP_COST(28);            % Fire Protection System
RDTE_Ass_Ref(10) = RDTE_COMP_COST(29);            % Flight Control System
RDTE_Ass_Ref(11) = RDTE_COMP_COST(30);            % Flight Compartment Furnishing System
RDTE_Ass_Ref(12) = RDTE_COMP_COST(31);           % Automatic Flight System
RDTE_Ass_Ref(13) = sum(RDTE_COMP_COST(32:43),"omitnan"); % Navigation System (sum of all Navigation System components)
RDTE_Ass_Ref(14) = sum(RDTE_COMP_COST(44:50),"omitnan"); % Comunication System (sum of all Communication System compoennts)
RDTE_Ass_Ref(15) = sum(RDTE_COMP_COST(51:59),"omitnan"); % Electrical System (sum of all Eletrical System compoentns)
RDTE_Ass_Ref(17) = RDTE_COMP_COST(60);         % Thermo Acustic Insulation System
RDTE_Ass_Ref(18) = RDTE_COMP_COST(61);         % Cockpit Crew System
RDTE_Ass_Ref(19) = RDTE_COMP_COST(62);         % Interior Integration System
RDTE_Ass_Ref(20) = RDTE_COMP_COST(63);         % Furnishing System
RDTE_Ass_Ref(21) = RDTE_COMP_COST(64);         % Fixed Oxygen System
RDTE_Ass_Ref(22) = RDTE_COMP_COST(65);         % Lighting System
RDTE_Ass_Ref(23) = RDTE_COMP_COST(66);         % Water System
RDTE_Ass_Ref(24) = RDTE_COMP_COST(67);         % Pax Seats System

PROD_Ass_Ref = zeros(26,1);     % Initializing PROD asselmblies' reference values array
PROD_Ass_Ref(1) = sum(PROD_COMP_COST(6:7),"omitnan");   % Landing Gear
PROD_Ass_Ref(4) = PROD_COMP_COST(20);            % APU System
PROD_Ass_Ref(5) = sum(PROD_COMP_COST(21:22),"omitnan"); % Hydraulic System
PROD_Ass_Ref(6) = sum(PROD_COMP_COST(23:24),"omitnan"); % Fuel System
PROD_Ass_Ref(7) = PROD_COMP_COST(25);            % ECS
PROD_Ass_Ref(8) = sum(PROD_COMP_COST(26:27),"omitnan"); % De-Icing System  
PROD_Ass_Ref(9) = PROD_COMP_COST(28);            % Fire Protection System
PROD_Ass_Ref(10) = PROD_COMP_COST(29);            % Flight Control System
PROD_Ass_Ref(11) = PROD_COMP_COST(30);            % Flight Compartment Furnishing System
PROD_Ass_Ref(12) = PROD_COMP_COST(31);           % Automatic Flight System
PROD_Ass_Ref(13) = sum(PROD_COMP_COST(32:43),"omitnan"); % Navigation System  
PROD_Ass_Ref(14) = sum(PROD_COMP_COST(44:50),"omitnan"); % Comunication System  
PROD_Ass_Ref(15) = sum(PROD_COMP_COST(51:59),"omitnan"); % Electrical System  
PROD_Ass_Ref(17) = PROD_COMP_COST(60);         % Thermo Acustic Insulation System
PROD_Ass_Ref(18) = PROD_COMP_COST(61);         % Cockpit Crew System
PROD_Ass_Ref(19) = PROD_COMP_COST(62);         % Interior Integration System
PROD_Ass_Ref(20) = PROD_COMP_COST(63);         % Furnishing System
PROD_Ass_Ref(21) = PROD_COMP_COST(64);         % Fixed Oxygen System
PROD_Ass_Ref(22) = PROD_COMP_COST(65);         % Lighting System
PROD_Ass_Ref(23) = PROD_COMP_COST(66);         % Water System
PROD_Ass_Ref(24) = PROD_COMP_COST(67);         % Pax Seats System

Ass1_Index = [1 4:15 17:24];    % Array containing the indexes of Level 1 Assemblies. Indexes 2,3,16,25,26 are left for Level 2 Assemblies and Total 

%% RDTE - PROD Level 1 Assemblies' Base Costs
for j=1:length(Ass1_Index)  % Only Level 1 Assemblies are considerd for this cycle
    i = Ass1_Index(j);
    switch RDTE_Type_ass(i)     % RDTE Cost estimation follows the same process and curve types described for Components. Specific CERs are used.
        case 'Poly1'
            RDTE_Ass_BaseCost(i,1) = RDTE_Ass_Ref(i)*RDTE_BaseCost_ass_coeff(i,1) + RDTE_BaseCost_ass_coeff(i,2);
        case 'Poly2'
            if RDTE_Ass_Ref(i) < X0_ass_RDTE(i)
                RDTE_Ass_BaseCost(i,1) =  RDTE_BaseCost_ass_coeff_P2(i,1)*(RDTE_Ass_Ref(i))^2 + RDTE_BaseCost_ass_coeff_P2(i,2)*RDTE_Ass_Ref(i) + RDTE_BaseCost_ass_coeff_P2(i,3);
            else
                RDTE_Ass_BaseCost(i,1) =  RDTE_BaseCost_ass_coeff_P2(i,1)*(X0_ass_RDTE(i))^2 + RDTE_BaseCost_ass_coeff_P2(i,2)*X0_ass_RDTE(i) + RDTE_BaseCost_ass_coeff_P2(i,3);
            end
        case 'P2-P1'
            if RDTE_Ass_Ref(i) < X0_ass_RDTE(i)
                RDTE_Ass_BaseCost(i,1) =  RDTE_BaseCost_ass_coeff_P2(i,1)*(RDTE_Ass_Ref(i))^2 + RDTE_BaseCost_ass_coeff_P2(i,2)*RDTE_Ass_Ref(i) + RDTE_BaseCost_ass_coeff_P2(i,3);
            else
                RDTE_Ass_BaseCost(i,1) = RDTE_Ass_Ref(i)*RDTE_BaseCost_ass_coeff(i,1) + RDTE_BaseCost_ass_coeff(i,2);
            end
    end
    switch PROD_Type_ass(i)     % PROD Cost estimation follows the same process and curve types described for Components. Specific CERs are used.
        case 'Poly1'
            PROD_Ass_BaseCost(i,1) = PROD_Ass_Ref(i)*PROD_BaseCost_ass_coeff(i,1) + PROD_BaseCost_ass_coeff(i,2);
        case 'Poly2'
            if PROD_Ass_Ref(i) < X0_ass_PROD(i)
                PROD_Ass_BaseCost(i,1) =  PROD_BaseCost_ass_coeff_P2(i,1)*(PROD_Ass_Ref(i))^2 + PROD_BaseCost_ass_coeff_P2(i,2)*PROD_Ass_Ref(i) + PROD_BaseCost_ass_coeff_P2(i,3);
            else
                PROD_Ass_BaseCost(i,1) =  PROD_BaseCost_ass_coeff_P2(i,1)*(X0_ass_PROD(i))^2 + PROD_BaseCost_ass_coeff_P2(i,2)*X0_ass_PROD(i) + PROD_BaseCost_ass_coeff_P2(i,3);
            end
        case 'P2-P1'
            if PROD_Ass_Ref(i) < X0_ass_PROD(i)
                PROD_Ass_BaseCost(i,1) =  PROD_BaseCost_ass_coeff_P2(i,1)*(PROD_Ass_Ref(i))^2 + PROD_BaseCost_ass_coeff_P2(i,2)*PROD_Ass_Ref(i) + PROD_BaseCost_ass_coeff_P2(i,3);
            else
                PROD_Ass_BaseCost(i,1) = PROD_Ass_Ref(i)*PROD_BaseCost_ass_coeff(i,1) + PROD_BaseCost_ass_coeff(i,2);
            end
    end
end

%% RDTE - PROD Level 1 Assemblies' Parameters Processing
% For each assembly, the multiplicative factor associated with each parameter that affects its cost is calculated
F_Ass_SysComp_RDTE = RDTE_Ass_Param(:,1).*(Ass_Parameters(:,1)).^2 + RDTE_Ass_Param(:,2).*Ass_Parameters(:,1) + RDTE_Ass_Param(:,3);    % System Complexity: F = a*EC^2 + b*EC + c
F_Ass_EngComp_RDTE = RDTE_Ass_Param(:,4).*(Ass_Parameters(:,2)).^2 + RDTE_Ass_Param(:,5).*Ass_Parameters(:,2) + RDTE_Ass_Param(:,6);    % Engineering Complexity for Assemblies: F = a*EC^2 + b*EC + c
F_Ass_Site_RDTE = RDTE_Ass_Site(:,1).*(General_Parameters(1)).^2 + RDTE_Ass_Site(:,2).*General_Parameters(1) + RDTE_Ass_Site(:,3);      % Multiple Site Development: F = a*EC^2 + b*EC + c

%% RDTE - PROD Level 1 Assemblies' Cost Output
% The estimated Base Cost for each Assembly is multiplied by the factors derived from the parameters
RDTE_ASS_COST = RDTE_Ass_BaseCost.*F_Ass_SysComp_RDTE(1:24).*F_Ass_EngComp_RDTE(1:24).*F_Ass_Site_RDTE;
PROD_ASS_COST = PROD_Ass_BaseCost;

%% RDTE - PROD Level 2 Assemblies' Reference Base Cost 
% Level 2 Assemblies are those in orange in the PBS
% The reference value for estimating Base Costs for each level 2 assembly is the 
% sum of the costs calculated for the level 1 assemblies from which it is formed.
RDTE_Ass_Ref(2) = sum(RDTE_COMP_COST(1:5),"omitnan") + RDTE_ASS_COST(1);   % STRUCTURE
RDTE_Ass_Ref(3) = sum(RDTE_COMP_COST(8:19),"omitnan");  % POWERPLANT
RDTE_Ass_Ref(16) = sum(RDTE_ASS_COST(4:15),"omitnan");  % SUBSYSTEMS
RDTE_Ass_Ref(25) = sum(RDTE_ASS_COST(17:24),"omitnan"); % FURNISHING

PROD_Ass_Ref(2) = sum(PROD_COMP_COST(1:5),"omitnan") + PROD_ASS_COST(1);   % STRUCTURE
PROD_Ass_Ref(3) = sum(PROD_COMP_COST(8:19),"omitnan");  % POWERPLANT
PROD_Ass_Ref(16) = sum(PROD_ASS_COST(4:15),"omitnan");  % SUBSYSTEMS
PROD_Ass_Ref(25) = sum(PROD_ASS_COST(17:24),"omitnan"); % FURNISHING

Ass2_Index = [2 3 16 25];       % Array containing the indexes of Level 2 Assemblies 

%% RDTE - PROD Level 2 Assemblies' Costs 
for j=1:length(Ass2_Index)      % Only Level 1 Assemblies are considerd for this cycle
    i = Ass2_Index(j);
    switch RDTE_Type_ass(i)     % RDTE Cost estimation follows the same process and curve types described for Components. Specific CERs are used.
        case 'Poly1'
            RDTE_Ass_BaseCost(i,1) = RDTE_Ass_Ref(i)*RDTE_BaseCost_ass_coeff(i,1) + RDTE_BaseCost_ass_coeff(i,2);
        case 'Poly2'
            if RDTE_Ass_Ref(i) < X0_ass_RDTE(i)
                RDTE_Ass_BaseCost(i,1) =  RDTE_BaseCost_ass_coeff_P2(i,1)*(RDTE_Ass_Ref(i))^2 + RDTE_BaseCost_ass_coeff_P2(i,2)*RDTE_Ass_Ref(i) + RDTE_BaseCost_ass_coeff_P2(i,3);
            else
                RDTE_Ass_BaseCost(i,1) =  RDTE_BaseCost_ass_coeff_P2(i,1)*(X0_ass_RDTE(i))^2 + RDTE_BaseCost_ass_coeff_P2(i,2)*X0_ass_RDTE(i) + RDTE_BaseCost_ass_coeff_P2(i,3);
            end
        case 'P2-P1'
            if RDTE_Ass_Ref(i) < X0_ass_RDTE(i)
                RDTE_Ass_BaseCost(i,1) =  RDTE_BaseCost_ass_coeff_P2(i,1)*(RDTE_Ass_Ref(i))^2 + RDTE_BaseCost_ass_coeff_P2(i,2)*RDTE_Ass_Ref(i) + RDTE_BaseCost_ass_coeff_P2(i,3);
            else
                RDTE_Ass_BaseCost(i,1) = RDTE_Ass_Ref(i)*RDTE_BaseCost_ass_coeff(i,1) + RDTE_BaseCost_ass_coeff(i,2);
            end
    end
    switch PROD_Type_ass(i)      % PROD Cost estimation follows the same process and curve types described for Components. Specific CERs are used.
        case 'Poly1'
            PROD_Ass_BaseCost(i,1) = PROD_Ass_Ref(i)*PROD_BaseCost_ass_coeff(i,1) + PROD_BaseCost_ass_coeff(i,2);
        case 'Poly2'
            if PROD_Ass_Ref(i) < X0_ass_PROD(i)
                PROD_Ass_BaseCost(i,1) =  PROD_BaseCost_ass_coeff_P2(i,1)*(PROD_Ass_Ref(i))^2 + PROD_BaseCost_ass_coeff_P2(i,2)*PROD_Ass_Ref(i) + PROD_BaseCost_ass_coeff_P2(i,3);
            else
                PROD_Ass_BaseCost(i,1) =  PROD_BaseCost_ass_coeff_P2(i,1)*(X0_ass_PROD(i))^2 + PROD_BaseCost_ass_coeff_P2(i,2)*X0_ass_PROD(i) + PROD_BaseCost_ass_coeff_P2(i,3);
            end
        case 'P2-P1'
            if PROD_Ass_Ref(i) < X0_ass_PROD(i)
                PROD_Ass_BaseCost(i,1) =  PROD_BaseCost_ass_coeff_P2(i,1)*(PROD_Ass_Ref(i))^2 + PROD_BaseCost_ass_coeff_P2(i,2)*PROD_Ass_Ref(i) + PROD_BaseCost_ass_coeff_P2(i,3);
            else
                PROD_Ass_BaseCost(i,1) = PROD_Ass_Ref(i)*PROD_BaseCost_ass_coeff(i,1) + PROD_BaseCost_ass_coeff(i,2);
            end
    end
    RDTE_ASS_COST(i,1) = RDTE_Ass_BaseCost(i)*F_Ass_SysComp_RDTE(i)*F_Ass_EngComp_RDTE(i);  % RDTE Costs are corrected with Parameters' Factors
    PROD_ASS_COST(i,1) = PROD_Ass_BaseCost(i,1);        % PROD Costs are not affected by parameters
end

%%% Powerplant for hybrid configurations: Poly2 for the entire range (no linearization)
if flag_powerplant==1
    PROD_Ass_BaseCost(3,1) =  PROD_BaseCost_ass_coeff_P2(3,1)*(PROD_Ass_Ref(3))^2 + PROD_BaseCost_ass_coeff_P2(3,2)*PROD_Ass_Ref(3) + PROD_BaseCost_ass_coeff_P2(3,3);
    PROD_ASS_COST(3,1) = PROD_Ass_BaseCost(3,1);
end

%% RDTE - PROD Total Aircraft with Integration Reference Base Cost
% The reference value for estimating Base Costs for Total Aircraft is the sum of the costs calculated for the 4 Level 2 Assemblies
RDTE_Ass_Ref(26) = RDTE_ASS_COST(2) + RDTE_ASS_COST(3) + RDTE_ASS_COST(16) + RDTE_ASS_COST(25);
PROD_Ass_Ref(26) = PROD_ASS_COST(2) + PROD_ASS_COST(3) + PROD_ASS_COST(16) + PROD_ASS_COST(25);

%% RDTE - PROD Total Aircraft with Integration Cost 
i=26;   % Index for Total Aircraft         
switch RDTE_Type_ass(i)     % RDTE Cost estimation follows the same process and curve types described for Components. Specific CER is used.
    case 'Poly1'
        RDTE_Ass_BaseCost(i,1) = RDTE_Ass_Ref(i)*RDTE_BaseCost_ass_coeff(i,1) + RDTE_BaseCost_ass_coeff(i,2);
    case 'Poly2'
        if RDTE_Ass_Ref(i) < X0_ass_RDTE(i)
            RDTE_Ass_BaseCost(i,1) =  RDTE_BaseCost_ass_coeff_P2(i,1)*(RDTE_Ass_Ref(i))^2 + RDTE_BaseCost_ass_coeff_P2(i,2)*RDTE_Ass_Ref(i) + RDTE_BaseCost_ass_coeff_P2(i,3);
        else
            RDTE_Ass_BaseCost(i,1) =  RDTE_BaseCost_ass_coeff_P2(i,1)*(X0_ass_RDTE(i))^2 + RDTE_BaseCost_ass_coeff_P2(i,2)*X0_ass_RDTE(i) + RDTE_BaseCost_ass_coeff_P2(i,3);
        end
    case 'P2-P1'
        if RDTE_Ass_Ref(i) < X0_ass_RDTE(i)
            RDTE_Ass_BaseCost(i,1) =  RDTE_BaseCost_ass_coeff_P2(i,1)*(RDTE_Ass_Ref(i))^2 + RDTE_BaseCost_ass_coeff_P2(i,2)*RDTE_Ass_Ref(i) + RDTE_BaseCost_ass_coeff_P2(i,3);
        else
            RDTE_Ass_BaseCost(i,1) = RDTE_Ass_Ref(i)*RDTE_BaseCost_ass_coeff(i,1) + RDTE_BaseCost_ass_coeff(i,2);
        end
end
switch PROD_Type_ass(i)     % PROD Cost estimation follows the same process and curve types described for Components. Specific CER is used.
    case 'Poly1'
        PROD_Ass_BaseCost(i,1) = PROD_Ass_Ref(i)*PROD_BaseCost_ass_coeff(i,1) + PROD_BaseCost_ass_coeff(i,2);
    case 'Poly2'
        if PROD_Ass_Ref(i) < X0_ass_PROD(i)
            PROD_Ass_BaseCost(i,1) =  PROD_BaseCost_ass_coeff_P2(i,1)*(PROD_Ass_Ref(i))^2 + PROD_BaseCost_ass_coeff_P2(i,2)*PROD_Ass_Ref(i) + PROD_BaseCost_ass_coeff_P2(i,3);
        else
            PROD_Ass_BaseCost(i,1) =  PROD_BaseCost_ass_coeff_P2(i,1)*(X0_ass_PROD(i))^2 + PROD_BaseCost_ass_coeff_P2(i,2)*X0_ass_PROD(i) + PROD_BaseCost_ass_coeff_P2(i,3);
        end
    case 'P2-P1'
        if PROD_Ass_Ref(i) < X0_ass_PROD(i)
            PROD_Ass_BaseCost(i,1) =  PROD_BaseCost_ass_coeff_P2(i,1)*(PROD_Ass_Ref(i))^2 + PROD_BaseCost_ass_coeff_P2(i,2)*PROD_Ass_Ref(i) + PROD_BaseCost_ass_coeff_P2(i,3);
        else
            PROD_Ass_BaseCost(i,1) = PROD_Ass_Ref(i)*PROD_BaseCost_ass_coeff(i,1) + PROD_BaseCost_ass_coeff(i,2);
        end
end

RDTE_ASS_COST(26,1) = RDTE_Ass_BaseCost(26)*F_Ass_SysComp_RDTE(26)*F_Ass_EngComp_RDTE(26);  % RDTE Cost is corrected with Parameters' Factors
PROD_ASS_COST(26,1) = PROD_Ass_BaseCost(26,1);        % PROD Cost is not affected by parameters

%% PROD Learning Curve
% Learning Curve effect is applied only on the cost increase from component(s) or sub-assembly to assembly

Assembly_increase = PROD_ASS_COST - PROD_Ass_Ref; % Increase from component(s) or sub-assemblies to assembly
a_ass_inc = Assembly_increase./146.2415;        % Increase cost for the first Assembly produced (Reference LC = 0.83) 

PROD_Ass_Inc_LC = zeros(length(a_ass_inc),1);   % Initializing LC corrected increase costs array

% Total PROD increase cost for 'n' units with 'LC_ass' slope is calculated for each Assembly
for i=1:length(PROD_ASS_COST)
    for n=1:n_prod
        PROD_Ass_Inc_LC(i) = PROD_Ass_Inc_LC(i) + a_ass_inc(i)*n^(log(LC_ass(i))/log(2));   % The cost for each unit in added to the total till n_prod is reached 
    end
end
PROD_ASS_COST_LC = PROD_Ass_Ref + PROD_Ass_Inc_LC;      % Total PROD Cost is calculated adding LC corrected increase costs to reference PROD costs

%% RDTE - PROD Total Program Reference Base Cost
% The reference value for estimating Base Costs for Total Program is the costs calculated for Total Aircraft
RDTE_Ass_Ref(27) = RDTE_ASS_COST(26);
PROD_Ass_Ref(27) = PROD_ASS_COST_LC(26);

%% RDTE - PROD Total Program Parameters Processing
% The multiplicative factor associated with each parameter that affects Total Program's cost is calculated
F_Prog_Team_RDTE = RDTE_Prog_Team(1)*(General_Parameters(2))^2 + RDTE_Prog_Team(2)*General_Parameters(2) + RDTE_Prog_Team(3);   % Management Team Complexity RDTE: F = a*MTC^2 + b*MTC + c
F_Prog_Org_RDTE = RDTE_Prog_Org(1)*General_Parameters(3)^RDTE_Prog_Org(2) + RDTE_Prog_Org(3);       % Mamagement Organizational Productivity: F = a*MOP^b + c
F_Prog_ProjComp_RDTE = RDTE_Prog_ProjComp(1)*General_Parameters(4) + RDTE_Prog_ProjComp(2);         % Project Complexity Factor RDTE: F = a*PCF + b

F_Prog_Team_PROD = PROD_Prog_Team(1)*(General_Parameters(2))^2 + PROD_Prog_Team(2)*General_Parameters(2) + PROD_Prog_Team(3);   % Management Team Complexity PROD: F = a*MTC^2 + b*MTC + c
F_Prog_ProjComp_PROD = PROD_Prog_ProjComp(1)*General_Parameters(4) + PROD_Prog_ProjComp(2);         % Project Complexity Factor PROD: F = a*PCF + b

%% RDTE - PROD Assemblies' Total Program Costs 
i=27;   % Index for Total Program
switch RDTE_Type_ass(i)     % RDTE Cost estimation follows the same process and curve types described for Components. Specific CER is used.
    case 'Poly1'
        RDTE_Ass_BaseCost(i,1) = RDTE_Ass_Ref(i)*RDTE_BaseCost_ass_coeff(i,1) + RDTE_BaseCost_ass_coeff(i,2);
    case 'Poly2'
        if RDTE_Ass_Ref(i) < X0_ass_RDTE(i)
            RDTE_Ass_BaseCost(i,1) =  RDTE_BaseCost_ass_coeff_P2(i,1)*(RDTE_Ass_Ref(i))^2 + RDTE_BaseCost_ass_coeff_P2(i,2)*RDTE_Ass_Ref(i) + RDTE_BaseCost_ass_coeff_P2(i,3);
        else
            RDTE_Ass_BaseCost(i,1) =  RDTE_BaseCost_ass_coeff_P2(i,1)*(X0_ass_RDTE(i))^2 + RDTE_BaseCost_ass_coeff_P2(i,2)*X0_ass_RDTE(i) + RDTE_BaseCost_ass_coeff_P2(i,3);
        end
    case 'P2-P1'
        if RDTE_Ass_Ref(i) < X0_ass_RDTE(i)
            RDTE_Ass_BaseCost(i,1) =  RDTE_BaseCost_ass_coeff_P2(i,1)*(RDTE_Ass_Ref(i))^2 + RDTE_BaseCost_ass_coeff_P2(i,2)*RDTE_Ass_Ref(i) + RDTE_BaseCost_ass_coeff_P2(i,3);
        else
            RDTE_Ass_BaseCost(i,1) = RDTE_Ass_Ref(i)*RDTE_BaseCost_ass_coeff(i,1) + RDTE_BaseCost_ass_coeff(i,2);
        end
end
switch PROD_Type_ass(i)     % PROD Cost estimation follows the same process and curve types described for Components. Specific CER is used.
    case 'Poly1'
        PROD_Ass_BaseCost(i,1) = PROD_Ass_Ref(i)*PROD_BaseCost_ass_coeff(i,1) + PROD_BaseCost_ass_coeff(i,2);
    case 'Poly2'
        if PROD_Ass_Ref(i) < X0_ass_PROD(i)
            PROD_Ass_BaseCost(i,1) =  PROD_BaseCost_ass_coeff_P2(i,1)*(PROD_Ass_Ref(i))^2 + PROD_BaseCost_ass_coeff_P2(i,2)*PROD_Ass_Ref(i) + PROD_BaseCost_ass_coeff_P2(i,3);
        else
            PROD_Ass_BaseCost(i,1) =  PROD_BaseCost_ass_coeff_P2(i,1)*(X0_ass_PROD(i))^2 + PROD_BaseCost_ass_coeff_P2(i,2)*X0_ass_PROD(i) + PROD_BaseCost_ass_coeff_P2(i,3);
        end
    case 'P2-P1'
        if PROD_Ass_Ref(i) < X0_ass_PROD(i)
            PROD_Ass_BaseCost(i,1) =  PROD_BaseCost_ass_coeff_P2(i,1)*(PROD_Ass_Ref(i))^2 + PROD_BaseCost_ass_coeff_P2(i,2)*PROD_Ass_Ref(i) + PROD_BaseCost_ass_coeff_P2(i,3);
        else
            PROD_Ass_BaseCost(i,1) = PROD_Ass_Ref(i)*PROD_BaseCost_ass_coeff(i,1) + PROD_BaseCost_ass_coeff(i,2);
        end
end

RDTE_ASS_COST(27,1) = RDTE_Ass_BaseCost(27)*F_Prog_Team_RDTE*F_Prog_Org_RDTE*F_Prog_ProjComp_RDTE;      % RDTE Cost is corrected with Parameters' Factors
PROD_ASS_COST_LC(27,1) = PROD_Ass_BaseCost(27)*F_Prog_Team_PROD*F_Prog_ProjComp_PROD;   % PROD Cost is corrected with Parameters' Factors

%% Costs Actualization, CEF e Currency Change
% All Costs caluclated are adjusted with inflaction 'USD_2023vs2017'
% The user can set an additional inflaction with 'CEF'
RDTE_COMP_COST_Act = RDTE_COMP_COST.*InflactionUSD_2023vs2017*CEF;
PROD_COMP_COST_LC_Act = PROD_COMP_COST_LC.*InflactionUSD_2023vs2017* CEF;
RDTE_ASS_COST_Act = RDTE_ASS_COST.*InflactionUSD_2023vs2017*CEF;
PROD_ASS_COST_LC_Act = PROD_ASS_COST_LC.*InflactionUSD_2023vs2017*CEF;

% If user selected 'EUR' as output currency, all cost are converted multiplying by the exchange factor entered
if isequal(Cur ,'EUR')
    RDTE_COMP_COST_Act = RDTE_COMP_COST.*Change;
    PROD_COMP_COST_LC_Act = PROD_COMP_COST_LC.*Change;
    RDTE_ASS_COST_Act = RDTE_ASS_COST.*Change;
    PROD_ASS_COST_LC_Act = PROD_ASS_COST_LC.*Change;
end

%% OUTPUT
fprintf('Writing RDTE + PROD output file...\n')

% RDTE and PROD costs are written in 2 separated worksheets, each one with the same formatting.
% 2 arrays are created reporting each component and assembly cost in the correct position, including blank rows.
RDTE_OUTPUT = NaN(102,2);   % Initializing RDTE Output array
PROD_OUTPUT = NaN(102,2);   % Initializing PROD Output array

Output_Pos_Comp = [1:7 12:23 27 29 30 32 33 35 37 38 40 42 44 46 48:59 61:67 69:77 82 84 86 88 90 92 94 96];    % Indexes of each component's cost
for i=1:length(RDTE_COMP_COST)
    j = Output_Pos_Comp(i);
    if RDTE_COMP_COST(i)>0      % Only the components actually planned (cost > 0) are written
        RDTE_OUTPUT(j,1) = RDTE_COMP_COST_Act(i)/n_prod;    % RDTE Cost per Unit
        RDTE_OUTPUT(j,2) = RDTE_COMP_COST_Act(i);           % Total RDTE Cost
    end
    if PROD_COMP_COST_LC(i)>0
        PROD_OUTPUT(j,1) = PROD_COMP_COST_LC_Act(i)/n_prod;     % PROD Cost per Unit (Average)
        PROD_OUTPUT(j,2) = PROD_COMP_COST_LC_Act(i);            % Total PROD cost
    end
end

Output_Pos_Ass = [8 9 24 28 31 34 36 39 41 43 45 47 60 68 78 79 83 85 87 89 91 93 95 97 98 100 102];    % Indexes of each component's cost
for i=1:27
    j = Output_Pos_Ass(i);
    if RDTE_ASS_COST(i)>0        % Only the assemblies actually planned (cost > 0) are written
        RDTE_OUTPUT(j,1) = RDTE_ASS_COST_Act(i)/n_prod;     % RDTE Cost per Unit
        RDTE_OUTPUT(j,2) = RDTE_ASS_COST_Act(i);            % Total RDTE Cost
    end
    if PROD_ASS_COST_LC(i)>0
        PROD_OUTPUT(j,1) = PROD_ASS_COST_LC_Act(i)/n_prod;  % PROD Cost per Unit (Average)
        PROD_OUTPUT(j,2) = PROD_ASS_COST_LC_Act(i);         % Total PROD cost
    end
end

% Setting actual currency symbol as requasted by the user
if isequal(Cur ,'EUR')
    Cur_Write(1,1) = "Average [€/Unit]";
    Cur_Write(1,2) = "Total [€]";
else
    Cur_Write(1,1) = "Average [$/Unit]";
    Cur_Write(1,2) = "Total [$]";
end

% Writing OUTPUT Arrays in the 'Results RDTE' and 'Results PROD' worksheets
xlswrite(file, cellstr(Cur_Write), 'Results RDTE', 'B3');
xlswrite(file, RDTE_OUTPUT(:,:), 'Results RDTE', 'B5');
xlswrite(file, cellstr(Cur_Write), 'Results PROD', 'B3');
xlswrite(file, PROD_OUTPUT(:,:), 'Results PROD', 'B5');

%%%%%%%%%%%%%%%%%%%%%
%% OPERATING COSTS %%
%%%%%%%%%%%%%%%%%%%%%
fprintf('\nProcessing Operating Costs...\n')

%% PRE_PROCESSING
% Defining variables for operating cost calculation
W = sum(Weight,'omitnan') - sum([Weight(18) Weight(42) Weight(43)],'omitnan');      % Total weight
[~,AircraftType] = xlsread(file, 'General Input', 'B3');    % Reading Aircraft Type from 'General Input' worksheet
AircraftType = string(AircraftType);
[~,EngineType] = xlsread(file, 'General Input', 'B5');      % Reading Engine Type from 'General Input' worksheet
EngineType = string(EngineType);
W_Engine = sum([Weight(8) Weight(10)],'omitnan');       % Total Engine weight
EngineNum = xlsread(file, 'General Input', 'B6');       % Reading Engine Number from 'General Input' worksheet
Profit = xlsread(file, 'General Input', 'B19');         % Reading Profit Margin from 'General Input' worksheet
Price = (RDTE_OUTPUT(:,1) + PROD_OUTPUT(:,1))*(1 + Profit);     % Calculating Aircraft Price = (RDTE_unit + PROD_unit)*(1 + Profit)

%% Operating - Input
% Importing Data from 'Operating' worksheet

%%% General Input
ACprice = xlsread(file,'Operating','C4');           % Aircraft price [$]
passenger_carried = xlsread(file,'Operating','C5'); % Number of passengers
n_e = xlsread(file,'Operating','C6');               % Number of engines
FH_Flight = xlsread(file,'Operating','C7');         % Flight Hours per flight [h]
BH_Flight = xlsread(file,'Operating','C8');         % Block Hours per flight [h]
NM_Flight = xlsread(file,'Operating','C9');         % Nautical miles per flight [nm]
num_trips_year = xlsread(file,'Operating','C10');   % Flights per year
fuel_consumption_kg = xlsread(file,'Operating','C11'); % Fuel mass per flight [kg]
CEF_2023 = xlsread(file,'Operating','C12');         % CEF wrt 2023
pinf = xlsread(file,'Operating','C13');             % Inflation rate [%]
region = xlsread(file,'Operating','C14');           % Selected world area

CEF_op = InflactionUSD_2023vs2017*CEF_2023;     % Total cost escalation wrt 2017

if ACprice > 0      % The user can enter the expected price for the aircraft 
else
    ACprice = Price(102);   % If no price is entered, it's calculated from RDTE & PROD costs plus Profit. 102 is the index for the Total Program price
end

%%% Depreciation
residual = xlsread(file,'Operating','C17');     % Residual value as percentage on acquisition cost [%]
period_dep = xlsread(file,'Operating','C18');   % Depreciation period [years]
airframe_spares = xlsread(file,'Operating','C19');  % Percentage of Airframe Spares cost referred to the acquisition cost of one aircraft [%]
engine_spares = xlsread(file,'Operating','C20');    % Percentage of Engine Spares cost referred to the acquisition cost of one engine [%]
EngineCost = sum(Price(12:19),"omitnan");   % Acquisition cost of the Engines
AirframeCost = ACprice - EngineCost;        % Acquisition cost of the Airframe

%%% Interest
period_int = xlsread(file,'Operating','C23');    % Interest period [years]
interest_rate = xlsread(file,'Operating','C24'); % Interest rate [%]
present_value = xlsread(file,'Operating','C25'); % Aircraft value at present time [$]
future_value = xlsread(file,'Operating','C26');  % Aircraft value at target year [$]

if present_value > 0         % The user can enter the expected value for the aircraft 
else
    present_value = Price(102);     % If no value is entered, it's setted equal to the acquisition price. 102 is the index for the Total Program price
end

%%% Cockpit Crew
LR1 = xlsread(file,'Operating','C29');      % Cockpit crew labour rate [$/BH]
n_crew = xlsread(file,'Operating','C30');   % Number of pilots

%%% Cabin Crew
LR2 = xlsread(file,'Operating','C33');      % Cabin crew labour rate [$/BH]
n_cabin = xlsread(file,'Operating','C34');  % Number of attendants

%%% Fuel expenses
fuel_price = xlsread(file,'Operating','C37'); % Market price for fuel [$/L]

%%% Noise
C_noise = xlsread(file,'Operating','C40');  % Noise rate [$/EPNdB]
L_appr = xlsread(file,'Operating','C41');   % Certified noise level - Approach [EPNdB]
L_fly = xlsread(file,'Operating','C42');    % Certified noise level - Flyover [EPNdB]
L_lat = xlsread(file,'Operating','C43');    % Certified noise level - Lateral [EPNdB]
Td = xlsread(file,'Operating','C44');   % Threshold - Departure [EPNdB]
Ta = xlsread(file,'Operating','C45');   % Threshold - Arrival [EPNdB]

%%% HC - NOx Emissions
n_y = xlsread(file,'Operating','C48'); % Year of estimation
n_a = xlsread(file,'Operating','C49'); % Number of airports where pollution charges are applied
eHC_LTO = xlsread(file,'Operating','C50');      % Hydrocarbon (HC) emissions [g]
e_NOx_LTO = xlsread(file,'Operating','C51');    % NOx emissions [g]

%%% CO2 Emissions
C_CO2 = xlsread(file,'Operating','C54');        % Cost for CO2 emission certificates per ton [$/ton]
p_CO2_free = xlsread(file,'Operating','C55');   % Fraction of CO2 emissions free of charge in percentage [%]

%%% Maintenance
fleetsize = xlsread(file,'Operating','C58');    % Number of aircraft in service
dayutil = xlsread(file,'Operating','C59');      % Flight Hour per day [FH/day]
fus = xlsread(file,'Operating','C60');          % Fuselage length [m]
agetype = xlsread(file,'Operating','C61');      % Number of years from entry into service of the selected aircraft model [years]
ACage = xlsread(file,'Operating','C62');        % Number of years of the selected aircraft [years]
ntyres = xlsread(file,'Operating','C63');       % Number of tyres
thrust = xlsread(file,'Operating','C64');       % Thrust per engine (turbofan) [N]
EnginePower = xlsread(file,'Operating','C65');  % Power per engine (turboprop) [kW]
EngineSpools = xlsread(file,'Operating','C66'); % Number of engine spools (power shaft excluded)
EngineSFC = xlsread(file,'Operating','C67');    % Engine SFC [lb/hp/h]

%%% Delay and Cancellation
delay_rate=xlsread(file,'Operating','C70');     % Annual delay rate [%]
avg_min_delay=xlsread(file,'Operating','C71');  % Average delay per flight [min]
cancrate=xlsread(file,'Operating','C72');       % Annual cancellation rate [%]

%%% Navigation and Landing
coeff_navigation=xlsread(file,'Operating','C75');   % Coefficient for Navigation fee [%/nm/lb]
K_landing=xlsread(file,'Operating','C76');          % Coefficient for Landing fee [$/ton]

%%% Standard IOC
pax_load = xlsread(file,'Operating','C81');     % Average aircraft filling rate [%]

%%% Direct Operating Costs for Hybrid-Electric Components
% Battery
Price_Battery = PROD_OUTPUT(20,1)*(1 + Profit); % Acquisition cost
energy_flight=xlsread(file,'Operating','C88');  % Required electric energy per flight [kWh]    
energy_cost=xlsread(file,'Operating','C89');    % Electric energy cost [$/kWh]
charging_eff=xlsread(file,'Operating','C90');   % Charging efficiency
bat_residual=xlsread(file,'Operating','C91');   % Residual value as percentrage on acquisition cost [%]
max_cycle=xlsread(file,'Operating','C92');  % Maximum cycle number for each battery
num_cycle=xlsread(file,'Operating','C93');  % Number of charhe/discharge cycles per flight

% Electric motor
Price_EM = Price(18);       % Acquisition cost
EM_MTBMA=xlsread(file,'Operating','C96');  % Mean Time Between MAintenance [BH]
EM_MAcost=xlsread(file,'Operating','C97'); % Maintenance cost as percentrage on acquisition cost [%]

% DC/AC inverter
Price_DCAC = Price(73);     % Acquisition cost
inv_MTBMA=xlsread(file,'Operating','C100');  % Mean Time Between MAintenance [BH]
inv_MAcost=xlsread(file,'Operating','C101'); % Maintenance cost as percentrage on acquisition cost [%]

% Thermal management system
Price_TMS = Price(21) + Price (77);     % Acquisition cost
TMS_MTBMA=xlsread(file,'Operating','C104');  % Mean Time Between MAintenance [BH]
TMS_MAcost=xlsread(file,'Operating','C105'); % Maintenance cost as percentrage on acquisition cost [%]

% Power cables
Price_Cables = Price(76);       % Acquisition cost
cab_MTBMA=xlsread(file,'Operating','C108');  % Mean Time Between MAintenance [BH]
cab_MAcost=xlsread(file,'Operating','C109'); % Maintenance cost as percentrage on acquisition cost [%]

% Electric generator
Price_Generator = Price(72);    % Acquisition cost
gen_MTBMA=xlsread(file,'Operating','C114');  % Mean Time Between MAintenance [BH]
gen_MAcost=xlsread(file,'Operating','C115'); % Maintenance cost as percentrage on acquisition cost [%]

% Rectifier (AC/DC)
Price_ACDC = Price(74);         % Acquisition cost
rect_MTBMA=xlsread(file,'Operating','C118');  % Mean Time Between MAintenance [BH]
rect_MAcost=xlsread(file,'Operating','C119'); % Maintenance cost as percentrage on acquisition cost [%]

% Fuel cell
Price_FC = Price(22);           % Acquisition cost
FC_MTBMA=xlsread(file,'Operating','C124');  % Mean Time Between MAintenance [BH]
FC_MAcost=xlsread(file,'Operating','C125'); % Maintenance cost as percentrage on acquisition cost [%]

% Liquid hydrogen
LH2_energy_flight=xlsread(file,'Operating','C128');     % Required mass per flight [kg]
LH2_cost=xlsread(file,'Operating','C129');          % LH2 cost [$/kg]
LH2_charging_eff=xlsread(file,'Operating','C130');  % LH2 charging efficiency

% LH2 tanks
Price_LH2tanks = Price(23);     % Acquisition cost
LH2_tank_MTBMA=xlsread(file,'Operating','C133');  % Mean Time Between MAintenance [BH]
LH2_tank_MAcost=xlsread(file,'Operating','C134'); % Maintenance cost as percentrage on acquisition cost [%]

% DC/DC converter
Price_DCDC = Price(75);         % Acquisition cost
conv_MTBMA=xlsread(file,'Operating','C137');  % Mean Time Between MAintenance [BH]
conv_MAcost=xlsread(file,'Operating','C138'); % Maintenance cost as percentrage on acquisition cost [%]

%% Costs Scaling Factors
coeff_1993=144.50;  % 1993 CPI 
coeff_1997=160.50;  % 1977 CPI 
coeff_2017=245.12;  % 2017 CPI 
coeff1=coeff_2017/coeff_1993;
coeff2=coeff_2017/coeff_1997;

%% Capital Costs %%

%%% Depreciation
COST.OPDIR.depreciation = ((1-residual)*(ACprice/period_dep)+airframe_spares*(AirframeCost/period_dep)+engine_spares*(EngineCost/period_dep)); % [$/Year]

%%% Interest
COST.OPDIR.interest = (present_value-future_value)*interest_rate/(1-(1+interest_rate)^(-period_int))-present_value/period_int; % [$/Year]

%%% Insurance
sector_length = NM_Flight * 1.852;  % Range in km
insurance_rate = 0.01;
COST.OPDIR.insurance = ACprice*insurance_rate; % [$/Year]

%% Crew Cost %%

%%% Cockpit crew cost
COST.OPDIR.CockpitCrew = BH_Flight*LR1*n_crew;     % [$/Trip]

%%% Cabin crew cost
COST.OPDIR.CabinCrew = BH_Flight*LR2*n_cabin;      % [$/Trip]

%%% Crew training %[$/flight]
COST.OPDIR.CrewTraining = (0.0184*(W/1000)^3-9.1399*(W/1000)^2+1621.7*(W/1000)+8734.3)/num_trips_year;     % [$/Trip]

%% Fuel and electricity expenses %%

%%% Fuel
rho_jet = 0.815;    % Jet-A1 density [kg/L]
fuel_consumption_litro = fuel_consumption_kg/rho_jet;   % Fuel consumption in L
fuel_cost_trip = fuel_price*fuel_consumption_litro;     % [$/trip]
COST.OPDIR.fuel = fuel_cost_trip*num_trips_year;              % [$/Year]

%%% Electric energy
if energy_flight>0      % Check if energy is required for flight
    COST.OPDIR.HYB.electric_energy_cost = energy_flight*energy_cost/charging_eff;   % [$/Trip]
else    % If the aircraft does not require electric energy, set cost to 0
    COST.OPDIR.HYB.electric_energy_cost = 0;
end

%%% LH2
if LH2_energy_flight>0  % Check if LH2 is required for flight
    COST.OPDIR.HYB.LH2_cost = LH2_energy_flight*LH2_cost/LH2_charging_eff;     % [$/Trip]
else    % If the aircraft does not require electric energy, set cost to 0
    COST.OPDIR.HYB.LH2_cost = 0;
end

%% Charges %%

%%% Landing Fees
Wt = W/1000;  % Aircraft's weight in tons
COST.OPDIR.landing_fees = K_landing*Wt*CEF_op*num_trips_year;    % [$/year]

%%% Navigation Charges  
Wp = W*2.20462;     % Aircraft's weight in pounds
COST.OPDIR.enroutecharge = coeff_navigation*NM_Flight*sqrt(Wp/1000)*coeff1*CEF_op*num_trips_year;     % [$/year]
 
%%% Noise
if region == 1      % Noise fee is planned requested exclusively for Europe
    exp1=(L_appr-Ta)/10;
    exp2=((L_fly+L_lat)/2-Td)/10;
    COST.OPDIR.noise = C_noise*((10^exp1)+(10^exp2))*num_trips_year;   % [$/year]
else
    fprintf('The requested region is currently not implemented\n')
    fprintf('DOC related to noise may not be available\n')
    COST.OPDIR.noise = 0;
end
       
%%% Pollutant Emission Charges
if region == 1      % Pollutant Emission fees are planned requested exclusively for Europe
    if eHC_LTO<=19.6
        COST.OPDIR.emiss_charg = (7.12*10^(-4)*(e_NOx_LTO*n_e*n_a)/25*(1+pinf)^(n_y-2011))*num_trips_year;   % [$/Year]
    else
        COST.OPDIR.emiss_charg = (2.12*10^(-5)*(e_NOx_LTO*eHC_LTO*n_e*n_a)/25*(1+pinf).^(n_y-2011))*num_trips_year;  % [$/Year] 
    end 
else
    fprintf('The requested region is currently not implemented\n')
    fprintf('DOC related to HC-NOx emissions may not be available\n')
    COST.OPDIR.emiss_charg = 0;
end   

%%% CO2 charges
if region == 1      % CO2 fees are planned requested exclusively for Europe
   COST.OPDIR.C_ETS = ((3.15*10^(-3)*fuel_consumption_kg*C_CO2*(17.6+0.7*(n_y-2010))*(1-(p_CO2_free/(100+2.5*(n_y-2005)))))/(64.4+3.1*(n_y-2010)))*num_trips_year;  % [$/Year]  
else
    fprintf('The requested region is currently not implemented\n')
    fprintf('DOC related to CO2 emissions may not be available\n')
    COST.OPDIR.C_ETS = 0;
end    

%% MAINTENANCE COST %%
coeff_2013 = 232.96;  % 2013 CPI
coeff3 = coeff_2017/coeff_2013;

fus = fus*3.28084;  % Fuselage length in feet

agetype = agetype*12;   % Type Age in months
ACprice = ACprice/1000000;  % Aircraft price in million
FHFC = FH_Flight;       % Flight Hours per Flight Cycle
maint_cost_drivers = [1 fleetsize dayutil FHFC fus ACprice agetype passenger_carried ACage ntyres n_e thrust]'; % Maint. cost drivers array definition

% Definition of the Coefficients Matrix
C=zeros(16,12);

% 1st column
C(1,1)=193.16;
C(2,1)=144.87;
C(3,1)=135.16;
C(4,1)=2.7564;
C(5,1)=5.1822;
C(6,1)=7.0216;
C(7,1)=2.2867;
C(8,1)=9.7101;
C(9,1)=4.8767;
C(10,1)=3.4695;
C(11,1)=1.9568;
C(12,1)=53.163;
C(13,1)=12.405;
C(14,1)=11.491;
C(15,1)=8.0316;
C(16,1)=5.181;

% 2nd colummn
C(1,2)=0.0107;
C(2,2)=0.008021;
C(5,2)=0.0013;
C(11,2)=0.0005;
C(14,2)=-0.0039;

% 3rd column
C(1,3)=-18.694;
C(2,3)=-14.02;
C(3,3)=-19.754;
C(5,3)=-0.1459;
C(6,3)=-0.3866;
C(7,3)=-0.74156;
C(8,3)=-0.75352;
C(9,3)=1.5254;
C(11,3)=-0.0551;
C(12,3)=-3.8567;
C(13,3)=-0.8999;
C(15,3)=0.77627;
C(16,3)=-0.75722;

% 4th column
C(1,4)=14.537;
C(2,4)=10.903;
C(4,4)=0.1178;
C(5,4)=1.0198;
C(7,4)=0.6649;
C(8,4)=0.4987;
C(10,4)=0.63766;
C(11,4)=0.3851;
C(12,4)=2.6676;
C(13,4)=0.62245;
C(14,4)=0.4836;
C(15,4)=3.9836;

% 5th column
C(7,5)=0.098302;

% 6th column
C(1,6)=0.88422;
C(2,6)=0.66317;
C(4,6)=0.0175;
C(6,6)=0.0423;
C(8,6)=0.050262;
C(12,6)=0.27299;
C(13,6)=0.063698;
C(14,6)=0.063;

% 7th column
C(1,7)=0.11926;
C(2,7)=0.089447;
C(3,7)=-0.01886;
C(4,7)=-0.0007;
C(5,7)=-0.006;
C(8,7)=0.001694;
C(9,7)=-0.01893;
C(10,7)=-0.00415;
C(11,7)=-0.0022;
C(12,7)=0.003441;
C(13,7)=0.000803;
C(14,7)=0.0097;
C(15,7)=-0.01826;
C(16,7)=-0.00072;

% 8th column
C(5,8)=0.0177;
C(6,8)=0.0003;
C(7,8)=0.005085;
C(8,8)=-0.00036;
C(9,8)=0.048375;
C(10,8)=0.012738;
C(11,8)=0.0067;
C(14,8)=0.0108;

% 9th column
C(1,9)=-1.972;
C(2,9)=-1.479;
C(4,9)=-0.0065;
C(9,9)=-0.01107;
C(14,9)=-0.2987;

% 10th column
C(12,10)=0.5725;
C(13,10)=0.13359;

% 11th column
C(3,11)=110.72;
C(16,11)=4.2443;

% 12th column
C(3,12)=0.005492;
C(16,12)=0.0002;

maint_costs = C*maint_cost_drivers*coeff3*CEF_op;   % Calculation of maintenance costs [%/FH]

% Assembling Maintanance Cost output based on region, aircraft type, engine type and systems included

if region==1   % EUROPE 
    maintenance_factor=1.597; % Factor used to compare maintenance costs between Europe and North America
end 

if region==2   % ASIA
    maintenance_factor=1.597; % Factor used to compare maintenance costs between Aisa and North America
end 

if region==3   % AFRICA
    maintenance_factor=1.597; % Factor used to compare maintenance costs between Africa and North America
end 

if region==4   % NORTH AMERICA
    maintenance_factor=1;  % Maintenance factor for North America
end 

if region==5   % SOUTH AMERICA
    maintenance_factor=1.597; % Factor used to compare maintenance costs between South America and North America
end 

if strcmp (AircraftType,'Regional')
    COST.OPDIR.maint_line = maint_costs(1,1)/1.513533450/maintenance_factor;    % Line maintenance cost for regional aircrafts [%/FH]
else
    COST.OPDIR.maint_line = maint_costs(1,1)/maintenance_factor;                % Line maintenance cost for other aircrafts [%/FH]
end

if strcmp (AircraftType,'Regional')
    COST.OPDIR.maint_base = maint_costs(2,1)/1.852997683/maintenance_factor;    % Base maintenance cost for regional aircrafts [%/FH]
else
    COST.OPDIR.maint_base = maint_costs(2,1/maintenance_factor);                % Base maintenance cost for other aircrafts [%/FH]
end

if strcmp (EngineType,'Turboprop')
    COST.OPDIR.maint_eng = ((((W_Engine*2.20462)*0.0283*(EngineSpools*EngineSFC*10000/(W_Engine*2.20462))^0.5259)*1.333333333)*EngineNum*1.082/0.887431723)/maintenance_factor;     % TP Engines maintenance cost [%/FH]
    COST.OPDIR.maint_prop = ((0.0033463669*(EnginePower/0.74569987)^1.0945240792)*EngineNum/1.052/1.171755013)/maintenance_factor;      % TP Propellers maintenance cost [%/FH]
else
    COST.OPDIR.maint_eng = maint_costs(3,1)+maint_costs(16,1)/maintenance_factor; % TF Engines + thrust reverser/feathering maintenance cost [%/FH]
    COST.OPDIR.maint_prop = 0;
end

if Quantity(31)>0   % Checks if Automatic Fligth System is included in the design
    if strcmp (AircraftType,'Regional')
        COST.OPDIR.maint_auto = maint_costs(4,1)/0.718630338/maintenance_factor;    % AFS maintenance cost for regional aircrafts [$/FH]
    else
        COST.OPDIR.maint_auto = maint_costs(4,1)/maintenance_factor;                % AFS maintenance cost for other aircrafts [$/FH]
    end
else        % If AFS is not included sets its maintenance cost to 0
    COST.OPDIR.maint_auto = 0;
end

COM = 0;    % Defining Communication System flag
for i=44:50
    if Quantity(i)>0    % Checks if any Comunication System Component is included in the design
        COM = 1;        % Sets Communication System flag to 1
    end
end

if COM==1       % Communication System included
    if strcmp (AircraftType,'Regional')
        COST.OPDIR.maint_com = (maint_costs(5,1)+maint_costs(11,1))/0.718630338/maintenance_factor; % Comm. System maintenance cost for regional aircrafts [$/FH]
    else
        COST.OPDIR.maint_com = maint_costs(5,1)+maint_costs(11,1)/maintenance_factor; % Comm. System maintenance cost for other aircrafts [$/FH]
    end
else        
    COST.OPDIR.maint_com = 0;   % If Comm. System is not included sets its maintenance cost to 0
end

EPGDS = 0;  % Defining Electrical System flag
for i=51:59
    if Quantity(i)>0    % Checks if any Electrical System Component is included in the design
        EPGDS = 1;      % Sets Electrical System flag to 1
    end
end

if EPGDS==1     % Electrical System included
    if strcmp (AircraftType,'Regional')
        COST.OPDIR.maint_ele = maint_costs(6,1)/0.718630338/maintenance_factor;     % Electrical System maintenance cost for regional aircrafts [$/FH]
    else
        COST.OPDIR.maint_ele = maint_costs(6,1)/maintenance_factor;     % Electrical System maintenance cost for other aircrafts [$/FH]
    end  
else
   COST.OPDIR.maint_ele = 0;    % If Electrical System is not included sets its maintenance cost to 0    
end

if Quantity(30)>0           % Checks if Flight Compartment Furnishing is included in the design
    if strcmp (AircraftType,'Regional')
        COST.OPDIR.maint_ef = maint_costs(7,1)/0.718630338/maintenance_factor;  % FCF maintenance cost for regional aircrafts [$/FH]
    else
        COST.OPDIR.maint_ef = maint_costs(7,1)/maintenance_factor;              % FCF maintenance cost for other aircrafts [$/FH]
    end  
else
    COST.OPDIR.maint_ef = 0;    % If FCF is not included sets its maintenance cost to 0
end

if Quantity(29)>0           % Checks if Flight Controls is included in the design 
    if strcmp (AircraftType,'Regional')
        COST.OPDIR.maint_fc = maint_costs(8,1)/0.718630338/maintenance_factor;  % Flight Controls maintenance cost for regional aircrafts [$/FH]
    else
        COST.OPDIR.maint_fc = maint_costs(8,1)/maintenance_factor;              % Flight Controls maintenance cost for other aircrafts [$/FH]
    end  
else
    COST.OPDIR.maint_fc = 0;    % If Flight Controls is not included sets its maintenance cost to 0
end

if Quantity(23)>0 || Quantity(24)>0         % Checks if Fuel System is included in the design 
    if strcmp (AircraftType,'Regional')
        COST.OPDIR.maint_fs = maint_costs(9,1)/0.718630338/maintenance_factor;  % Fuel System maintenance cost for regional aircrafts [$/FH]
    else
        COST.OPDIR.maint_fs = maint_costs(9,1)/maintenance_factor;              % Fuel System maintenance cost for other aircrafts [$/FH]
    end 
else
    COST.OPDIR.maint_fs = 0;    % If Fuel System is not included sets its maintenance cost to 0    
end

if Quantity(21)>0 || Quantity(22)>0         % Checks if Hydraulic System is included in the design   
    if strcmp (AircraftType,'Regional')
        COST.OPDIR.maint_hp = maint_costs(10,1)/0.718630338/maintenance_factor;     % Hydraulic System maintenance cost for regional aircrafts [$/FH]
    else
        COST.OPDIR.maint_hp = maint_costs(10,1)/maintenance_factor;                 % Hydraulic System maintenance cost for other aircrafts [$/FH]
    end
else
   COST.OPDIR.maint_hp = 0;     % If Hydraulic System is not included sets its maintenance cost to 0       
end

if Quantity(6)>0 || Quantity(7)>0           % Checks if Landing Gear is included in the design 
    if strcmp (AircraftType,'Regional')
        COST.OPDIR.maint_wb = maint_costs(12,1)/2.237563418/maintenance_factor;     % Wheels and Brakes maintenance cost for regional aircrafts [$/FH]
        COST.OPDIR.maint_lg = maint_costs(13,1)/0.447198022/maintenance_factor;     % Landing Gear maintenance cost for regional aircrafts [$/FH]
    else
        COST.OPDIR.maint_wb = maint_costs(12,1)/maintenance_factor;         % Wheels and Brakes maintenance cost for other aircrafts [$/FH]
        COST.OPDIR.maint_lg = maint_costs(13,1)/maintenance_factor;         % Landing Gear maintenance cost for other aircrafts [$/FH]
    end
else
    COST.OPDIR.maint_wb = 0;    % If Wheels and Brakes are not included sets their maintenance cost to 0
    COST.OPDIR.maint_lg = 0;    % If Landing Gear is not included sets its maintenance cost to 0
end

NAV = 0;        % Defining Navigation System flag
for i=32:43
    if Quantity(i)>0    % Checks if any Navigation System Component is included in the design
        NAV = 1;        % Sets Navigation System flag to 1
    end
end

if NAV==1       % Navigation System included
    if strcmp (AircraftType,'Regional')
        COST.OPDIR.maint_nav = maint_costs(14,1)/0.718630338/maintenance_factor;    % Navigation System maintenance cost for regional aircrafts [$/FH]
    else
        COST.OPDIR.maint_nav = maint_costs(14,1)/maintenance_factor;                % Navigation System maintenance cost for other aircrafts [$/FH]
    end
else
    COST.OPDIR.maint_nav = 0;   % If Navigation System is not included sets its maintenance cost to 0   
end

if Quantity(20)>0           % Checks if APU is included in the design
    COST.OPDIR.maint_apu = maint_costs(15,1)/maintenance_factor;    % APU maintenance cost [$/FH]
else
    COST.OPDIR.maint_apu = 0;   % If APU is not included sets its maintenance cost to 0       
end

AIRFRAME = COST.OPDIR.maint_line+COST.OPDIR.maint_base+COST.OPDIR.maint_wb+COST.OPDIR.maint_lg+COST.OPDIR.maint_apu+COST.OPDIR.maint_hp+...
       +COST.OPDIR.maint_fs+COST.OPDIR.maint_fc+COST.OPDIR.maint_ef+COST.OPDIR.maint_auto+...
       +COST.OPDIR.maint_nav+COST.OPDIR.maint_com+COST.OPDIR.maint_ele+COST.OPDIR.maint_prop;   % Total Airframe maintenance cost [$/FH]

ENGINE = COST.OPDIR.maint_eng;  % Engine maintenance cost [$/FH]
 
%% Maintenance for Hybrid Components
if max_cycle>0     % Checks if Battery is included in the design
    COST.OPDIR.HYB.maint_bat = Price_Battery*(1-bat_residual)/max_cycle*num_cycle;  % Battery maintenance cost [$/Trip]
else
    COST.OPDIR.HYB.maint_bat = 0;   % If Battery is not included sets its maintenance cost to 0
end

if EM_MTBMA>0      % Checks if Electric Motors are included in the design    
    COST.OPDIR.HYB.maint_EM = Price_EM*EM_MAcost/EM_MTBMA;    % Electric Motors maintenance cost [$/BH]
else
    COST.OPDIR.HYB.maint_EM = 0;    % If Electric Motors are not included sets their maintenance cost to 0
end

if inv_MTBMA>0     % Checks if Inverters (DC/AC) are included in the design
    COST.OPDIR.HYB.maint_inv = Price_DCAC*inv_MAcost/inv_MTBMA;    % Inverters (DC/AC) maintenance cost [$/BH]
else
    COST.OPDIR.HYB.maint_inv = 0;   % If Inverters (DC/AC) are not included sets their maintenance cost to 0
end
    
if TMS_MTBMA>0     % Checks if Thermal Management System is included in the design
    COST.OPDIR.HYB.maint_TMS = Price_TMS*TMS_MAcost/TMS_MTBMA;    % Thermal Management System maintenance cost [$/BH]
else
    COST.OPDIR.HYB.maint_TMS = 0;   % If Thermal Management System is not included sets its maintenance cost to 0
end

if cab_MTBMA>0     % Checks if Electrical Cables are included in the design
    COST.OPDIR.HYB.maint_cab = Price_Cables*cab_MAcost/cab_MTBMA;    % Electrical Cables maintenance cost [$/BH]
else
    COST.OPDIR.HYB.maint_cab = 0;   % If Electrical Cables are not included sets their maintenance cost to 0
end

if gen_MTBMA>0     % Checks if Generators are included in the design
    COST.OPDIR.HYB.maint_gen = Price_Generator*gen_MAcost/gen_MTBMA;    % Generators maintenance cost [$/BH]
else
    COST.OPDIR.HYB.maint_gen = 0;   % If Generators are not included sets their maintenance cost to 0
end

if rect_MTBMA>0    % Checks if Rectifiers (AC/DC) are included in the design
    COST.OPDIR.HYB.maint_rect = Price_ACDC*rect_MAcost/rect_MTBMA;    % Rectifiers (AC/DC) maintenance cost [$/BH]
else
    COST.OPDIR.HYB.maint_rect = 0;  % If Rectifiers (AC/DC) are not included sets their maintenance cost to 0
end

if FC_MTBMA>0      % Checks if Fuel Cells are included in the design
    COST.OPDIR.HYB.maint_FC = Price_FC*FC_MAcost/FC_MTBMA;    % Fuel Cells maintenance cost [$/BH]
else
    COST.OPDIR.HYB.maint_FC = 0;    % If Fuel Cells are not included sets their maintenance cost to 0
end
    
if LH2_tank_MTBMA>0    % Checks if LH2 Tanks are included in the design
    COST.OPDIR.HYB.maint_LH2_tanks = Price_LH2tanks*LH2_tank_MAcost/LH2_tank_MTBMA;    % LH2 Tanks maintenance cost [$/BH]
else
    COST.OPDIR.HYB.maint_LH2_tanks = 0;     % If LH2 Tanks are not included sets their maintenance cost to 0
end

if conv_MTBMA>0    % Checks if Converters (DC/DC) is included in the design
    COST.OPDIR.HYB.maint_conv = Price_DCDC*conv_MAcost/conv_MTBMA;    % Converters (DC/DC) maintenance cost [$/BH]
else
    COST.OPDIR.HYB.maint_conv = 0;  % If Converters (DC/DC) are not included sets their maintenance cost to 0
end

ENGINE = ENGINE+COST.OPDIR.HYB.maint_bat/FH_Flight+(COST.OPDIR.HYB.maint_EM+COST.OPDIR.HYB.maint_inv+COST.OPDIR.HYB.maint_rect+ ...
    COST.OPDIR.HYB.maint_conv+COST.OPDIR.HYB.maint_TMS+COST.OPDIR.HYB.maint_cab+COST.OPDIR.HYB.maint_gen+ ...
    COST.OPDIR.HYB.maint_FC+COST.OPDIR.HYB.maint_LH2_tanks)*BH_Flight/FH_Flight;    % Total Engine maintenance cost [$/FH]

COST.OPDIR.DMC = AIRFRAME + ENGINE;    % Direct Maintenance Cost
COST.OPDIR.TOTAL = COST.OPDIR.DMC/(60/100);     % Total Maintenance Cost
COST.OPDIR.Burden = 0.4*COST.OPDIR.TOTAL;       % Maintenance Burden

%% Operational Interruptions %%

%%% Delay cost
coeff_2004 = 188.90;  % 2004 CPI
coeff5 = coeff_2017/coeff_2004;
change_euro_doll_2004 = 1.23;   % EUR -> US$ change in 2004

d_cost_gate = ((0.0044*passenger_carried)+0.3134)*coeff5*CEF_op*change_euro_doll_2004; % Gate Delay cost per minute [$/min]
d_cost_airb = ((0.1022*passenger_carried)-1.6724)*coeff5*CEF_op*change_euro_doll_2004; % Airborne Delay cost per minute[$/min]
X = (d_cost_gate+d_cost_airb)/2;    % Mean delay cost per minute [$/min]
avg_cost = X*avg_min_delay;     % Cost for each delay
COST.OPDIR.delay = (num_trips_year*delay_rate)*avg_cost;  % Delay cost [$/Year]
       
%%% Cancellation cost 
if region == 1      % Cancelllation Cost are planned requested exclusively for Europe
    coeff_2014 = 236.74;    % 2014 CPI
    coeffcanc = coeff_2017/coeff_2014;      
    avg_value = 124*passenger_carried*coeffcanc*CEF_op;  % Average cost per cancelled flight
    COST.OPDIR.cancellation = (num_trips_year*cancrate)*avg_value;  % Cancellation cost [$/Year]  
else
    fprintf('The requested region is currently not implemented\n')
    fprintf('DOC related to CO2 emissions may not be available\n')
    COST.OPDIR.cancellation = 0;  
end 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% INDIRECT OPERATING COSTS%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%    

%% Traffic Service 
coeff_2012 = 229.59;  % 2012 CPI
coeff6 = coeff_2017/coeff_2012;  
COST.OPIND.trafficservice_cost = 15*passenger_carried*pax_load*num_trips_year*coeff6*CEF_op;  % [$/Year]

%% Passenger Service
COST.OPIND.passervice_cost = 0.0079*passenger_carried*NM_Flight*num_trips_year*coeff6*CEF_op;   % [$/Year]

%% Reservations and Sales
COST.OPIND.RS_cost = 0.0076*passenger_carried*NM_Flight*num_trips_year*coeff6*CEF_op;   % [$/Year]

%% General and Administrative
COST.OPIND.GAcost = 0.0072*passenger_carried*NM_Flight*num_trips_year*coeff6*CEF_op;     % [$/Year]
     
%% Station and Ground
COST.OPIND.SGcost = 0.0092*passenger_carried*NM_Flight*num_trips_year*coeff6*CEF_op;  % [$/Year]

COST_IOC_TOTAL = COST.OPIND.trafficservice_cost+COST.OPIND.passervice_cost+COST.OPIND.RS_cost+COST.OPIND.GAcost+COST.OPIND.SGcost;  % Total IOC [$/Year]

%% POST-PROCESSING

%%% Results in $/Year
COST.OPDIR_OUTYEAR.depreciation = COST.OPDIR.depreciation;
COST.OPDIR_OUTYEAR.interest = COST.OPDIR.interest;
COST.OPDIR_OUTYEAR.insurance = COST.OPDIR.insurance;
COST.OPDIR_OUTYEAR.CockpitCrew = COST.OPDIR.CockpitCrew*num_trips_year;
COST.OPDIR_OUTYEAR.CabinCrew = COST.OPDIR.CabinCrew*num_trips_year;
COST.OPDIR_OUTYEAR.CrewTraining = COST.OPDIR.CrewTraining*num_trips_year;
COST.OPDIR_OUTYEAR.fuel = COST.OPDIR.fuel;
COST.OPDIR_OUTYEAR.electric_energy = COST.OPDIR.HYB.electric_energy_cost*num_trips_year;
COST.OPDIR_OUTYEAR.LH2 = COST.OPDIR.HYB.LH2_cost*num_trips_year;
COST.OPDIR_OUTYEAR.enroutecharge = COST.OPDIR.enroutecharge;
COST.OPDIR_OUTYEAR.landing_fees = COST.OPDIR.landing_fees;
COST.OPDIR_OUTYEAR.noise = COST.OPDIR.noise;
COST.OPDIR_OUTYEAR.emiss_charg = COST.OPDIR.emiss_charg;
COST.OPDIR_OUTYEAR.C_ETS = COST.OPDIR.C_ETS;
COST.OPDIR_OUTYEAR.maint_line = COST.OPDIR.maint_line*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_base = COST.OPDIR.maint_base*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_eng = COST.OPDIR.maint_eng*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_EM = COST.OPDIR.HYB.maint_EM*BH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_gen = COST.OPDIR.HYB.maint_gen*BH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_prop = COST.OPDIR.maint_prop*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_bat = COST.OPDIR.HYB.maint_bat*num_trips_year;
COST.OPDIR_OUTYEAR.maint_FC = (COST.OPDIR.HYB.maint_FC+COST.OPDIR.HYB.maint_LH2_tanks)*BH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_TMS = COST.OPDIR.HYB.maint_TMS*BH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_auto = COST.OPDIR.maint_auto*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_com = COST.OPDIR.maint_com*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_ele = COST.OPDIR.maint_ele*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_conv = (COST.OPDIR.HYB.maint_inv+COST.OPDIR.HYB.maint_rect+COST.OPDIR.HYB.maint_conv)*BH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_cab = COST.OPDIR.HYB.maint_cab*BH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_ef = COST.OPDIR.maint_ef*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_fc = COST.OPDIR.maint_fc*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_fs = COST.OPDIR.maint_fs*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_hp = COST.OPDIR.maint_hp*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_wb = COST.OPDIR.maint_wb*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_lg = COST.OPDIR.maint_lg*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_nav = COST.OPDIR.maint_nav*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.maint_apu = COST.OPDIR.maint_apu*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.DMC = COST.OPDIR.DMC*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.AIRFRAME = AIRFRAME*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.ENGINE = ENGINE*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.TOTAL = COST.OPDIR.TOTAL*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.Burden = COST.OPDIR.Burden*FH_Flight*num_trips_year;
COST.OPDIR_OUTYEAR.delay = COST.OPDIR.delay;
COST.OPDIR_OUTYEAR.cancellation = COST.OPDIR.cancellation;

COST_DOC_TOTAL_OUTYEAR = COST.OPDIR_OUTYEAR.depreciation+COST.OPDIR_OUTYEAR.interest+COST.OPDIR_OUTYEAR.insurance+COST.OPDIR_OUTYEAR.CockpitCrew+...
    COST.OPDIR_OUTYEAR.CabinCrew+COST.OPDIR_OUTYEAR.CrewTraining+COST.OPDIR_OUTYEAR.fuel+COST.OPDIR_OUTYEAR.electric_energy+COST.OPDIR_OUTYEAR.LH2+COST.OPDIR_OUTYEAR.enroutecharge+COST.OPDIR_OUTYEAR.landing_fees+...
    COST.OPDIR_OUTYEAR.noise+COST.OPDIR_OUTYEAR.emiss_charg+COST.OPDIR_OUTYEAR.C_ETS+...
    COST.OPDIR_OUTYEAR.TOTAL+COST.OPDIR_OUTYEAR.delay+COST.OPDIR_OUTYEAR.cancellation; % Total DOC

COST_COC_TOTAL_OUTYEAR = COST_DOC_TOTAL_OUTYEAR-(COST.OPDIR_OUTYEAR.depreciation+COST.OPDIR_OUTYEAR.interest+COST.OPDIR_OUTYEAR.insurance+COST.OPDIR_OUTYEAR.delay+COST.OPDIR_OUTYEAR.cancellation);    % Total COC

%%% Results in $/Flight ($/Trip)
COST.OPDIR_OUTTRIP.depreciation = COST.OPDIR.depreciation/num_trips_year;
COST.OPDIR_OUTTRIP.interest = COST.OPDIR.interest/num_trips_year;
COST.OPDIR_OUTTRIP.insurance = COST.OPDIR.insurance/num_trips_year;
COST.OPDIR_OUTTRIP.CockpitCrew = COST.OPDIR.CockpitCrew;
COST.OPDIR_OUTTRIP.CabinCrew = COST.OPDIR.CabinCrew;
COST.OPDIR_OUTTRIP.CrewTraining=COST.OPDIR.CrewTraining;
COST.OPDIR_OUTTRIP.fuel = COST.OPDIR.fuel/num_trips_year;
COST.OPDIR_OUTTRIP.electric_energy = COST.OPDIR.HYB.electric_energy_cost;
COST.OPDIR_OUTTRIP.LH2 = COST.OPDIR.HYB.LH2_cost;
COST.OPDIR_OUTTRIP.enroutecharge = COST.OPDIR.enroutecharge/num_trips_year;
COST.OPDIR_OUTTRIP.landing_fees = COST.OPDIR.landing_fees/num_trips_year;
COST.OPDIR_OUTTRIP.noise = COST.OPDIR.noise/num_trips_year;
COST.OPDIR_OUTTRIP.emiss_charg = COST.OPDIR.emiss_charg/num_trips_year;
COST.OPDIR_OUTTRIP.C_ETS = COST.OPDIR.C_ETS/num_trips_year;
COST.OPDIR_OUTTRIP.maint_line = COST.OPDIR.maint_line*FH_Flight;
COST.OPDIR_OUTTRIP.maint_base = COST.OPDIR.maint_base*FH_Flight;
COST.OPDIR_OUTTRIP.maint_eng = COST.OPDIR.maint_eng*FH_Flight;
COST.OPDIR_OUTTRIP.maint_EM = COST.OPDIR.HYB.maint_EM*BH_Flight;
COST.OPDIR_OUTTRIP.maint_gen = COST.OPDIR.HYB.maint_gen*BH_Flight;
COST.OPDIR_OUTTRIP.maint_prop = COST.OPDIR.maint_prop*FH_Flight;
COST.OPDIR_OUTTRIP.maint_bat = COST.OPDIR.HYB.maint_bat;
COST.OPDIR_OUTTRIP.maint_FC = (COST.OPDIR.HYB.maint_FC+COST.OPDIR.HYB.maint_LH2_tanks)*BH_Flight;
COST.OPDIR_OUTTRIP.maint_TMS = COST.OPDIR.HYB.maint_TMS*BH_Flight;
COST.OPDIR_OUTTRIP.maint_auto = COST.OPDIR.maint_auto*FH_Flight;
COST.OPDIR_OUTTRIP.maint_com = COST.OPDIR.maint_com*FH_Flight;
COST.OPDIR_OUTTRIP.maint_ele = COST.OPDIR.maint_ele*FH_Flight;
COST.OPDIR_OUTTRIP.maint_conv = (COST.OPDIR.HYB.maint_inv+COST.OPDIR.HYB.maint_rect+COST.OPDIR.HYB.maint_conv)*BH_Flight;
COST.OPDIR_OUTTRIP.maint_cab = COST.OPDIR.HYB.maint_cab*BH_Flight;
COST.OPDIR_OUTTRIP.maint_ef = COST.OPDIR.maint_ef*FH_Flight;
COST.OPDIR_OUTTRIP.maint_fc = COST.OPDIR.maint_fc*FH_Flight;
COST.OPDIR_OUTTRIP.maint_fs = COST.OPDIR.maint_fs*FH_Flight;
COST.OPDIR_OUTTRIP.maint_hp = COST.OPDIR.maint_hp*FH_Flight;
COST.OPDIR_OUTTRIP.maint_wb = COST.OPDIR.maint_wb*FH_Flight;
COST.OPDIR_OUTTRIP.maint_lg = COST.OPDIR.maint_lg*FH_Flight;
COST.OPDIR_OUTTRIP.maint_nav = COST.OPDIR.maint_nav*FH_Flight;
COST.OPDIR_OUTTRIP.maint_apu = COST.OPDIR.maint_apu*FH_Flight;
COST.OPDIR_OUTTRIP.DMC = COST.OPDIR.DMC*FH_Flight;
COST.OPDIR_OUTTRIP.AIRFRAME = AIRFRAME*FH_Flight;
COST.OPDIR_OUTTRIP.ENGINE = ENGINE*FH_Flight;
COST.OPDIR_OUTTRIP.TOTAL = COST.OPDIR.TOTAL*FH_Flight;
COST.OPDIR_OUTTRIP.Burden = COST.OPDIR.Burden*FH_Flight;
COST.OPDIR_OUTTRIP.delay = COST.OPDIR.delay/num_trips_year;
COST.OPDIR_OUTTRIP.cancellation = COST.OPDIR.cancellation/num_trips_year;

COST_DOC_TOTAL_OUTTRIP = COST.OPDIR_OUTTRIP.depreciation+COST.OPDIR_OUTTRIP.interest+COST.OPDIR_OUTTRIP.insurance+COST.OPDIR_OUTTRIP.CockpitCrew+...
    COST.OPDIR_OUTTRIP.CabinCrew+COST.OPDIR_OUTTRIP.CrewTraining+COST.OPDIR_OUTTRIP.fuel+COST.OPDIR_OUTTRIP.electric_energy+COST.OPDIR_OUTTRIP.LH2+COST.OPDIR_OUTTRIP.enroutecharge+COST.OPDIR_OUTTRIP.landing_fees+...
    COST.OPDIR_OUTTRIP.noise+COST.OPDIR_OUTTRIP.emiss_charg+COST.OPDIR_OUTTRIP.C_ETS+...
    COST.OPDIR_OUTTRIP.TOTAL+COST.OPDIR_OUTTRIP.delay+COST.OPDIR_OUTTRIP.cancellation; % Total DOC

COST_COC_TOTAL_OUTTRIP = COST_DOC_TOTAL_OUTTRIP-(COST.OPDIR_OUTTRIP.depreciation+COST.OPDIR_OUTTRIP.interest+COST.OPDIR_OUTTRIP.insurance+COST.OPDIR_OUTTRIP.delay+COST.OPDIR_OUTTRIP.cancellation);    % Total COC

% Results in $/FH
COST.OPDIR_OUTFH.depreciation = COST.OPDIR.depreciation/num_trips_year/FH_Flight;
COST.OPDIR_OUTFH.interest = COST.OPDIR.interest/num_trips_year/FH_Flight;
COST.OPDIR_OUTFH.insurance = COST.OPDIR.insurance/num_trips_year/FH_Flight;
COST.OPDIR_OUTFH.CockpitCrew = COST.OPDIR.CockpitCrew/FH_Flight;
COST.OPDIR_OUTFH.CabinCrew = COST.OPDIR.CabinCrew/FH_Flight;
COST.OPDIR_OUTFH.CrewTraining=COST.OPDIR.CrewTraining/FH_Flight;
COST.OPDIR_OUTFH.fuel = COST.OPDIR.fuel/num_trips_year/FH_Flight;
COST.OPDIR_OUTFH.electric_energy = COST.OPDIR.HYB.electric_energy_cost/FH_Flight;
COST.OPDIR_OUTFH.LH2 = COST.OPDIR.HYB.LH2_cost/FH_Flight;
COST.OPDIR_OUTFH.enroutecharge = COST.OPDIR.enroutecharge/num_trips_year/FH_Flight;
COST.OPDIR_OUTFH.landing_fees = COST.OPDIR.landing_fees/num_trips_year/FH_Flight;
COST.OPDIR_OUTFH.noise = COST.OPDIR.noise/num_trips_year/FH_Flight;
COST.OPDIR_OUTFH.emiss_charg = COST.OPDIR.emiss_charg/num_trips_year/FH_Flight;
COST.OPDIR_OUTFH.C_ETS = COST.OPDIR.C_ETS/num_trips_year/FH_Flight;
COST.OPDIR_OUTFH.maint_line = COST.OPDIR.maint_line;
COST.OPDIR_OUTFH.maint_base = COST.OPDIR.maint_base;
COST.OPDIR_OUTFH.maint_eng = COST.OPDIR.maint_eng;
COST.OPDIR_OUTFH.maint_EM = COST.OPDIR.HYB.maint_EM*BH_Flight/FH_Flight;
COST.OPDIR_OUTFH.maint_gen = COST.OPDIR.HYB.maint_gen*BH_Flight/FH_Flight;
COST.OPDIR_OUTFH.maint_prop = COST.OPDIR.maint_prop;
COST.OPDIR_OUTFH.maint_bat = COST.OPDIR.HYB.maint_bat/FH_Flight;
COST.OPDIR_OUTFH.maint_FC = (COST.OPDIR.HYB.maint_FC+COST.OPDIR.HYB.maint_LH2_tanks)*BH_Flight/FH_Flight;
COST.OPDIR_OUTFH.maint_TMS = COST.OPDIR.HYB.maint_TMS*BH_Flight/FH_Flight;
COST.OPDIR_OUTFH.maint_auto = COST.OPDIR.maint_auto;
COST.OPDIR_OUTFH.maint_com = COST.OPDIR.maint_com;
COST.OPDIR_OUTFH.maint_ele = COST.OPDIR.maint_ele;
COST.OPDIR_OUTFH.maint_conv = (COST.OPDIR.HYB.maint_inv+COST.OPDIR.HYB.maint_rect+COST.OPDIR.HYB.maint_conv)*BH_Flight/FH_Flight;
COST.OPDIR_OUTFH.maint_cab = COST.OPDIR.HYB.maint_cab*BH_Flight/FH_Flight;
COST.OPDIR_OUTFH.maint_ef = COST.OPDIR.maint_ef;
COST.OPDIR_OUTFH.maint_fc = COST.OPDIR.maint_fc;
COST.OPDIR_OUTFH.maint_fs = COST.OPDIR.maint_fs;
COST.OPDIR_OUTFH.maint_hp = COST.OPDIR.maint_hp;
COST.OPDIR_OUTFH.maint_wb = COST.OPDIR.maint_wb;
COST.OPDIR_OUTFH.maint_lg = COST.OPDIR.maint_lg;
COST.OPDIR_OUTFH.maint_nav = COST.OPDIR.maint_nav;
COST.OPDIR_OUTFH.maint_apu = COST.OPDIR.maint_apu;
COST.OPDIR_OUTFH.DMC = COST.OPDIR.DMC;
COST.OPDIR_OUTFH.AIRFRAME = AIRFRAME;
COST.OPDIR_OUTFH.ENGINE = ENGINE;
COST.OPDIR_OUTFH.TOTAL = COST.OPDIR.TOTAL;
COST.OPDIR_OUTFH.Burden = COST.OPDIR.Burden;
COST.OPDIR_OUTFH.delay = COST.OPDIR.delay/num_trips_year/FH_Flight;
COST.OPDIR_OUTFH.cancellation = COST.OPDIR.cancellation/num_trips_year/FH_Flight;

COST_DOC_TOTAL_OUTFH = COST.OPDIR_OUTFH.depreciation+COST.OPDIR_OUTFH.interest+COST.OPDIR_OUTFH.insurance+COST.OPDIR_OUTFH.CockpitCrew+...
    COST.OPDIR_OUTFH.CabinCrew+COST.OPDIR_OUTFH.CrewTraining+COST.OPDIR_OUTFH.fuel+COST.OPDIR_OUTFH.electric_energy+COST.OPDIR_OUTFH.LH2+COST.OPDIR_OUTFH.enroutecharge+COST.OPDIR_OUTFH.landing_fees+...
    COST.OPDIR_OUTFH.noise+COST.OPDIR_OUTFH.emiss_charg+COST.OPDIR_OUTFH.C_ETS+...
    COST.OPDIR_OUTFH.TOTAL+COST.OPDIR_OUTFH.delay+COST.OPDIR_OUTFH.cancellation; % Total DOC

COST_COC_TOTAL_OUTFH = COST_DOC_TOTAL_OUTFH-(COST.OPDIR_OUTFH.depreciation+COST.OPDIR_OUTFH.interest+COST.OPDIR_OUTFH.insurance+COST.OPDIR_OUTFH.delay+COST.OPDIR_OUTFH.cancellation);  % Total COC

% Results in $/BH
COST.OPDIR_OUTBH.depreciation = COST.OPDIR.depreciation/num_trips_year/BH_Flight;
COST.OPDIR_OUTBH.interest = COST.OPDIR.interest/num_trips_year/BH_Flight;
COST.OPDIR_OUTBH.insurance = COST.OPDIR.insurance/num_trips_year/BH_Flight;
COST.OPDIR_OUTBH.CockpitCrew = COST.OPDIR.CockpitCrew/BH_Flight;
COST.OPDIR_OUTBH.CabinCrew = COST.OPDIR.CabinCrew/BH_Flight;
COST.OPDIR_OUTBH.CrewTraining=COST.OPDIR.CrewTraining/BH_Flight;
COST.OPDIR_OUTBH.fuel = COST.OPDIR.fuel/num_trips_year/BH_Flight;
COST.OPDIR_OUTBH.electric_energy = COST.OPDIR.HYB.electric_energy_cost/BH_Flight;
COST.OPDIR_OUTBH.LH2 = COST.OPDIR.HYB.LH2_cost/BH_Flight;
COST.OPDIR_OUTBH.enroutecharge = COST.OPDIR.enroutecharge/num_trips_year/BH_Flight;
COST.OPDIR_OUTBH.landing_fees = COST.OPDIR.landing_fees/num_trips_year/BH_Flight;
COST.OPDIR_OUTBH.noise = COST.OPDIR.noise/num_trips_year/BH_Flight;
COST.OPDIR_OUTBH.emiss_charg = COST.OPDIR.emiss_charg/num_trips_year/BH_Flight;
COST.OPDIR_OUTBH.C_ETS = COST.OPDIR.C_ETS/num_trips_year/BH_Flight;
COST.OPDIR_OUTBH.maint_line = COST.OPDIR.maint_line*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.maint_base = COST.OPDIR.maint_base*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.maint_eng = COST.OPDIR.maint_eng*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.maint_EM = COST.OPDIR.HYB.maint_EM;
COST.OPDIR_OUTBH.maint_gen = COST.OPDIR.HYB.maint_gen;
COST.OPDIR_OUTBH.maint_prop = COST.OPDIR.maint_prop*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.maint_bat = COST.OPDIR.HYB.maint_bat/BH_Flight;
COST.OPDIR_OUTBH.maint_FC = COST.OPDIR.HYB.maint_FC+COST.OPDIR.HYB.maint_LH2_tanks;
COST.OPDIR_OUTBH.maint_TMS = COST.OPDIR.HYB.maint_TMS;
COST.OPDIR_OUTBH.maint_auto = COST.OPDIR.maint_auto*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.maint_com = COST.OPDIR.maint_com*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.maint_ele = COST.OPDIR.maint_ele*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.maint_conv = COST.OPDIR.HYB.maint_inv+COST.OPDIR.HYB.maint_rect+COST.OPDIR.HYB.maint_conv;
COST.OPDIR_OUTBH.maint_cab = COST.OPDIR.HYB.maint_cab;
COST.OPDIR_OUTBH.maint_ef = COST.OPDIR.maint_ef*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.maint_fc = COST.OPDIR.maint_fc*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.maint_fs = COST.OPDIR.maint_fs*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.maint_hp = COST.OPDIR.maint_hp*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.maint_wb = COST.OPDIR.maint_wb*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.maint_lg = COST.OPDIR.maint_lg*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.maint_nav = COST.OPDIR.maint_nav*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.maint_apu = COST.OPDIR.maint_apu*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.DMC = COST.OPDIR.DMC*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.AIRFRAME = AIRFRAME*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.ENGINE = ENGINE*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.TOTAL = COST.OPDIR.TOTAL*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.Burden = COST.OPDIR.Burden*FH_Flight/BH_Flight;
COST.OPDIR_OUTBH.delay = COST.OPDIR.delay/num_trips_year/BH_Flight;
COST.OPDIR_OUTBH.cancellation = COST.OPDIR.cancellation/num_trips_year/BH_Flight;

COST_DOC_TOTAL_OUTBH = COST.OPDIR_OUTBH.depreciation+COST.OPDIR_OUTBH.interest+COST.OPDIR_OUTBH.insurance+...
COST.OPDIR_OUTBH.CockpitCrew+COST.OPDIR_OUTBH.CabinCrew+COST.OPDIR_OUTBH.CrewTraining+COST.OPDIR_OUTBH.fuel+COST.OPDIR_OUTBH.electric_energy+...
    COST.OPDIR_OUTBH.LH2+COST.OPDIR_OUTBH.enroutecharge+COST.OPDIR_OUTBH.landing_fees+COST.OPDIR_OUTBH.noise+COST.OPDIR_OUTBH.emiss_charg+...
    COST.OPDIR_OUTBH.C_ETS+COST.OPDIR_OUTBH.TOTAL+COST.OPDIR_OUTBH.delay+COST.OPDIR_OUTBH.cancellation; % Total DOC

COST_COC_TOTAL_OUTBH = COST_DOC_TOTAL_OUTBH-(COST.OPDIR_OUTBH.depreciation+COST.OPDIR_OUTBH.interest+COST.OPDIR_OUTBH.insurance+COST.OPDIR_OUTBH.delay+COST.OPDIR_OUTBH.cancellation);  % Total COC

% Results in $/NM
COST.OPDIR_OUTNM.depreciation = COST.OPDIR.depreciation/num_trips_year/NM_Flight;
COST.OPDIR_OUTNM.interest = COST.OPDIR.interest/num_trips_year/NM_Flight;
COST.OPDIR_OUTNM.insurance = COST.OPDIR.insurance/num_trips_year/NM_Flight;
COST.OPDIR_OUTNM.CockpitCrew = COST.OPDIR.CockpitCrew/NM_Flight;
COST.OPDIR_OUTNM.CabinCrew = COST.OPDIR.CabinCrew/NM_Flight;
COST.OPDIR_OUTNM.CrewTraining=COST.OPDIR.CrewTraining/NM_Flight;
COST.OPDIR_OUTNM.fuel = COST.OPDIR.fuel/num_trips_year/NM_Flight;
COST.OPDIR_OUTNM.electric_energy = COST.OPDIR.HYB.electric_energy_cost/NM_Flight;
COST.OPDIR_OUTNM.LH2 = COST.OPDIR.HYB.LH2_cost/NM_Flight;
COST.OPDIR_OUTNM.enroutecharge = COST.OPDIR.enroutecharge/num_trips_year/NM_Flight;
COST.OPDIR_OUTNM.landing_fees = COST.OPDIR.landing_fees/num_trips_year/NM_Flight;
COST.OPDIR_OUTNM.noise = COST.OPDIR.noise/num_trips_year/NM_Flight;
COST.OPDIR_OUTNM.emiss_charg = COST.OPDIR.emiss_charg/num_trips_year/NM_Flight;
COST.OPDIR_OUTNM.C_ETS = COST.OPDIR.C_ETS/num_trips_year/NM_Flight;
COST.OPDIR_OUTNM.maint_line = COST.OPDIR.maint_line*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_base = COST.OPDIR.maint_base*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_eng = COST.OPDIR.maint_eng*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_EM = COST.OPDIR.HYB.maint_EM*BH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_gen = COST.OPDIR.HYB.maint_gen*BH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_prop = COST.OPDIR.maint_prop*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_bat = COST.OPDIR.HYB.maint_bat/NM_Flight;
COST.OPDIR_OUTNM.maint_FC = (COST.OPDIR.HYB.maint_FC+COST.OPDIR.HYB.maint_LH2_tanks)*BH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_TMS = COST.OPDIR.HYB.maint_TMS*BH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_auto = COST.OPDIR.maint_auto*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_com = COST.OPDIR.maint_com*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_ele = COST.OPDIR.maint_ele*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_conv = (COST.OPDIR.HYB.maint_inv+COST.OPDIR.HYB.maint_rect+COST.OPDIR.HYB.maint_conv)*BH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_cab = COST.OPDIR.HYB.maint_cab*BH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_ef = COST.OPDIR.maint_ef*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_fc = COST.OPDIR.maint_fc*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_fs = COST.OPDIR.maint_fs*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_hp = COST.OPDIR.maint_hp*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_wb = COST.OPDIR.maint_wb*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_lg = COST.OPDIR.maint_lg*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_nav = COST.OPDIR.maint_nav*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.maint_apu = COST.OPDIR.maint_apu*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.DMC = COST.OPDIR.DMC*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.AIRFRAME = AIRFRAME*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.ENGINE = ENGINE*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.TOTAL = COST.OPDIR.TOTAL*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.Burden = COST.OPDIR.Burden*FH_Flight/NM_Flight;
COST.OPDIR_OUTNM.delay = COST.OPDIR.delay/num_trips_year/NM_Flight;
COST.OPDIR_OUTNM.cancellation = COST.OPDIR.cancellation/num_trips_year/NM_Flight;

COST_DOC_TOTAL_OUTNM = COST.OPDIR_OUTNM.depreciation+COST.OPDIR_OUTNM.interest+COST.OPDIR_OUTNM.insurance+COST.OPDIR_OUTNM.CockpitCrew+...
    COST.OPDIR_OUTNM.CabinCrew+COST.OPDIR_OUTNM.CrewTraining+COST.OPDIR_OUTNM.fuel+COST.OPDIR_OUTNM.electric_energy+COST.OPDIR_OUTNM.LH2+COST.OPDIR_OUTNM.enroutecharge+COST.OPDIR_OUTNM.landing_fees+...
    COST.OPDIR_OUTNM.noise+COST.OPDIR_OUTNM.emiss_charg+COST.OPDIR_OUTNM.C_ETS+...
    COST.OPDIR_OUTNM.TOTAL+COST.OPDIR_OUTNM.delay+COST.OPDIR_OUTNM.cancellation; % Total DOC

COST_COC_TOTAL_OUTNM = COST_DOC_TOTAL_OUTNM-(COST.OPDIR_OUTNM.depreciation+COST.OPDIR_OUTNM.interest+COST.OPDIR_OUTNM.insurance+COST.OPDIR_OUTNM.delay+COST.OPDIR_OUTNM.cancellation);  % Total COC

% Results in $/Seat
COST.OPDIR_OUTSEAT.depreciation = COST.OPDIR.depreciation/num_trips_year/passenger_carried;
COST.OPDIR_OUTSEAT.interest = COST.OPDIR.interest/num_trips_year/passenger_carried;
COST.OPDIR_OUTSEAT.insurance = COST.OPDIR.insurance/num_trips_year/passenger_carried;
COST.OPDIR_OUTSEAT.CockpitCrew = COST.OPDIR.CockpitCrew/passenger_carried;
COST.OPDIR_OUTSEAT.CabinCrew = COST.OPDIR.CabinCrew/passenger_carried;
COST.OPDIR_OUTSEAT.CrewTraining=COST.OPDIR.CrewTraining/passenger_carried;
COST.OPDIR_OUTSEAT.fuel = COST.OPDIR.fuel/num_trips_year/passenger_carried;
COST.OPDIR_OUTSEAT.electric_energy = COST.OPDIR.HYB.electric_energy_cost/passenger_carried;
COST.OPDIR_OUTSEAT.LH2 = COST.OPDIR.HYB.LH2_cost/passenger_carried;
COST.OPDIR_OUTSEAT.enroutecharge = COST.OPDIR.enroutecharge/num_trips_year/passenger_carried;
COST.OPDIR_OUTSEAT.landing_fees = COST.OPDIR.landing_fees/num_trips_year/passenger_carried;
COST.OPDIR_OUTSEAT.noise = COST.OPDIR.noise/num_trips_year/passenger_carried;
COST.OPDIR_OUTSEAT.emiss_charg = COST.OPDIR.emiss_charg/num_trips_year/passenger_carried;
COST.OPDIR_OUTSEAT.C_ETS = COST.OPDIR.C_ETS/num_trips_year/passenger_carried;
COST.OPDIR_OUTSEAT.maint_line = COST.OPDIR.maint_line*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_base = COST.OPDIR.maint_base*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_eng = COST.OPDIR.maint_eng*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_EM = COST.OPDIR.HYB.maint_EM*BH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_gen = COST.OPDIR.HYB.maint_gen*BH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_prop = COST.OPDIR.maint_prop*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_bat = COST.OPDIR.HYB.maint_bat/passenger_carried;
COST.OPDIR_OUTSEAT.maint_FC = (COST.OPDIR.HYB.maint_FC+COST.OPDIR.HYB.maint_LH2_tanks)*BH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_TMS = COST.OPDIR.HYB.maint_TMS*BH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_auto = COST.OPDIR.maint_auto*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_com = COST.OPDIR.maint_com*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_ele = COST.OPDIR.maint_ele*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_conv = (COST.OPDIR.HYB.maint_inv+COST.OPDIR.HYB.maint_rect+COST.OPDIR.HYB.maint_conv)*BH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_cab = COST.OPDIR.HYB.maint_cab*BH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_ef = COST.OPDIR.maint_ef*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_fc = COST.OPDIR.maint_fc*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_fs = COST.OPDIR.maint_fs*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_hp = COST.OPDIR.maint_hp*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_wb = COST.OPDIR.maint_wb*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_lg = COST.OPDIR.maint_lg*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_nav = COST.OPDIR.maint_nav*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.maint_apu = COST.OPDIR.maint_apu*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.DMC = COST.OPDIR.DMC*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.AIRFRAME = AIRFRAME*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.ENGINE = ENGINE*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.TOTAL = COST.OPDIR.TOTAL*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.Burden = COST.OPDIR.Burden*FH_Flight/passenger_carried;
COST.OPDIR_OUTSEAT.delay = COST.OPDIR.delay/num_trips_year/passenger_carried;
COST.OPDIR_OUTSEAT.cancellation = COST.OPDIR.cancellation/num_trips_year/passenger_carried;

COST_DOC_TOTAL_OUTSEAT = COST.OPDIR_OUTSEAT.depreciation+COST.OPDIR_OUTSEAT.interest+COST.OPDIR_OUTSEAT.insurance+COST.OPDIR_OUTSEAT.CockpitCrew+...
    COST.OPDIR_OUTSEAT.CabinCrew+COST.OPDIR_OUTSEAT.CrewTraining+COST.OPDIR_OUTSEAT.fuel+COST.OPDIR_OUTSEAT.electric_energy+COST.OPDIR_OUTSEAT.LH2+COST.OPDIR_OUTSEAT.enroutecharge+COST.OPDIR_OUTSEAT.landing_fees+...
    COST.OPDIR_OUTSEAT.noise+COST.OPDIR_OUTSEAT.emiss_charg+COST.OPDIR_OUTSEAT.C_ETS+...
    COST.OPDIR_OUTSEAT.TOTAL+COST.OPDIR_OUTSEAT.delay+COST.OPDIR_OUTSEAT.cancellation; % Total DOC

COST_COC_TOTAL_OUTSEAT = COST_DOC_TOTAL_OUTSEAT-(COST.OPDIR_OUTSEAT.depreciation+COST.OPDIR_OUTSEAT.interest+COST.OPDIR_OUTSEAT.insurance+COST.OPDIR_OUTSEAT.delay+COST.OPDIR_OUTSEAT.cancellation);    % Total COC

% Results in $/(NM*Seat)
COST.OPDIR_OUTNMSEAT.depreciation = COST.OPDIR.depreciation/num_trips_year/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.interest = COST.OPDIR.interest/num_trips_year/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.insurance = COST.OPDIR.insurance/num_trips_year/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.CockpitCrew = COST.OPDIR.CockpitCrew/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.CabinCrew = COST.OPDIR.CabinCrew/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.CrewTraining=COST.OPDIR.CrewTraining/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.fuel = COST.OPDIR.fuel/num_trips_year/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.electric_energy = COST.OPDIR.HYB.electric_energy_cost/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.LH2 = COST.OPDIR.HYB.LH2_cost/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.enroutecharge = COST.OPDIR.enroutecharge/num_trips_year/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.landing_fees = COST.OPDIR.landing_fees/num_trips_year/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.noise = COST.OPDIR.noise/num_trips_year/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.emiss_charg = COST.OPDIR.emiss_charg/num_trips_year/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.C_ETS = COST.OPDIR.C_ETS/num_trips_year/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_line = COST.OPDIR.maint_line*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_base = COST.OPDIR.maint_base*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_eng = COST.OPDIR.maint_eng*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_EM = COST.OPDIR.HYB.maint_EM*BH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_gen = COST.OPDIR.HYB.maint_gen*BH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_prop = COST.OPDIR.maint_prop*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_bat = COST.OPDIR.HYB.maint_bat/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_FC = (COST.OPDIR.HYB.maint_FC+COST.OPDIR.HYB.maint_LH2_tanks)*BH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_TMS = COST.OPDIR.HYB.maint_TMS*BH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_auto = COST.OPDIR.maint_auto*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_com = COST.OPDIR.maint_com*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_ele = COST.OPDIR.maint_ele*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_conv = (COST.OPDIR.HYB.maint_inv+COST.OPDIR.HYB.maint_rect+COST.OPDIR.HYB.maint_conv)*BH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_cab = COST.OPDIR.HYB.maint_cab*BH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_ef = COST.OPDIR.maint_ef*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_fc = COST.OPDIR.maint_fc*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_fs = COST.OPDIR.maint_fs*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_hp = COST.OPDIR.maint_hp*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_wb = COST.OPDIR.maint_wb*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_lg = COST.OPDIR.maint_lg*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_nav = COST.OPDIR.maint_nav*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.maint_apu = COST.OPDIR.maint_apu*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.DMC = COST.OPDIR.DMC*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.AIRFRAME = AIRFRAME*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.ENGINE = ENGINE*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.TOTAL = COST.OPDIR.TOTAL*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.Burden = COST.OPDIR.Burden*FH_Flight/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.delay = COST.OPDIR.delay/num_trips_year/NM_Flight/passenger_carried;
COST.OPDIR_OUTNMSEAT.cancellation = COST.OPDIR.cancellation/num_trips_year/NM_Flight/passenger_carried;

COST_DOC_TOTAL_OUTNMSEAT = COST.OPDIR_OUTNMSEAT.depreciation+COST.OPDIR_OUTNMSEAT.interest+COST.OPDIR_OUTNMSEAT.insurance+COST.OPDIR_OUTNMSEAT.CockpitCrew+...
    COST.OPDIR_OUTNMSEAT.CabinCrew+COST.OPDIR_OUTNMSEAT.CrewTraining+COST.OPDIR_OUTNMSEAT.fuel+COST.OPDIR_OUTNMSEAT.electric_energy+COST.OPDIR_OUTNMSEAT.LH2+COST.OPDIR_OUTNMSEAT.enroutecharge+COST.OPDIR_OUTNMSEAT.landing_fees+...
    COST.OPDIR_OUTNMSEAT.noise+COST.OPDIR_OUTNMSEAT.emiss_charg+COST.OPDIR_OUTNMSEAT.C_ETS+...
    COST.OPDIR_OUTNMSEAT.TOTAL+COST.OPDIR_OUTNMSEAT.delay+COST.OPDIR_OUTNMSEAT.cancellation; % Total DOC

COST_COC_TOTAL_OUTNMSEAT = COST_DOC_TOTAL_OUTNMSEAT-(COST.OPDIR_OUTNMSEAT.depreciation+COST.OPDIR_OUTNMSEAT.interest+COST.OPDIR_OUTNMSEAT.insurance+COST.OPDIR_OUTNMSEAT.delay+COST.OPDIR_OUTNMSEAT.cancellation);  % Total COC

%% Writing Results to Output file
fprintf('Writing OPERATING output file...\n')

% Converting results structures to cell arrays
OPDIR_OUTYEAR = struct2cell(COST.OPDIR_OUTYEAR);
OPDIR_OUTTRIP = struct2cell(COST.OPDIR_OUTTRIP);
OPDIR_OUTFH = struct2cell(COST.OPDIR_OUTFH);
OPDIR_OUTBH = struct2cell(COST.OPDIR_OUTBH);
OPDIR_OUTNM = struct2cell(COST.OPDIR_OUTNM);
OPDIR_OUTSEAT = struct2cell(COST.OPDIR_OUTSEAT);
OPDIR_OUTNMSEAT = struct2cell(COST.OPDIR_OUTNMSEAT);

OPIND = struct2cell(COST.OPIND);

% If user selected 'EUR' as output currency, all cost are converted by multiplying by the exchange factor entered
if isequal(Cur ,'EUR')
    OPDIR_OUTYEAR = OPDIR_OUTYEAR.*Change;
    OPDIR_OUTTRIP = OPDIR_OUTTRIP.*Change;
    OPDIR_OUTFH = OPDIR_OUTFH.*Change;
    OPDIR_OUTBH = OPDIR_OUTBH.*Change;
    OPDIR_OUTNM = OPDIR_OUTNM .*Change;
    OPDIR_OUTSEAT = OPDIR_OUTSEAT.*Change;
    OPDIR_OUTNMSEAT  = OPDIR_OUTNMSEAT .*Change;

    COST_DOC_TOTAL_OUTYEAR = COST_DOC_TOTAL_OUTYEAR.*Change;
    COST_DOC_TOTAL_OUTTRIP = COST_DOC_TOTAL_OUTTRIP.*Change;
    COST_DOC_TOTAL_OUTFH = COST_DOC_TOTAL_OUTFH.*Change;
    COST_DOC_TOTAL_OUTBH = COST_DOC_TOTAL_OUTBH.*Change;
    COST_DOC_TOTAL_OUTNM = COST_DOC_TOTAL_OUTNM.*Change;
    COST_DOC_TOTAL_OUTSEAT = COST_DOC_TOTAL_OUTSEAT.*Change;
    COST_DOC_TOTAL_OUTNMSEAT = COST_DOC_TOTAL_OUTNMSEAT.*Change;

    COST_COC_TOTAL_OUTYEAR = COST_COC_TOTAL_OUTYEAR.*Change;
    COST_COC_TOTAL_OUTTRIP = COST_COC_TOTAL_OUTTRIP.*Change;
    COST_COC_TOTAL_OUTFH = COST_COC_TOTAL_OUTFH.*Change;
    COST_COC_TOTAL_OUTBH = COST_COC_TOTAL_OUTBH.*Change;
    COST_COC_TOTAL_OUTNM = COST_COC_TOTAL_OUTNM.*Change;
    COST_COC_TOTAL_OUTSEAT = COST_COC_TOTAL_OUTSEAT.*Change;
    COST_COC_TOTAL_OUTNMSEAT = COST_COC_TOTAL_OUTNMSEAT.*Change;

    OPIND = OPIND.*Change;
    COST_IOC_TOTAL = COST_IOC_TOTAL.*Change;
end

% Defining starting cell for each result array
xlRange_OPDIR_OUTYEAR = 'C4';
xlRange_OPDIR_OUTTRIP = 'D4';
xlRange_OPDIR_OUTFH = 'E4';
xlRange_OPDIR_OUTBH = 'F4';
xlRange_OPDIR_OUTNM = 'G4';
xlRange_OPDIR_OUTSEAT = 'H4';
xlRange_OPDIR_OUTNMSEAT = 'I4';

xlRange_DOC_TOTAL_OUTYEAR = 'C47';
xlRange_DOC_TOTAL_OUTTRIP = 'D47';
xlRange_DOC_TOTAL_OUTFH = 'E47';
xlRange_DOC_TOTAL_OUTBH = 'F47';
xlRange_DOC_TOTAL_OUTNM = 'G47';
xlRange_DOC_TOTAL_OUTSEAT = 'H47';
xlRange_DOC_TOTAL_OUTNMSEAT = 'I47';

xlRange_COC_TOTAL_OUTYEAR = 'C48';
xlRange_COC_TOTAL_OUTTRIP = 'D48';
xlRange_COC_TOTAL_OUTFH = 'E48';
xlRange_COC_TOTAL_OUTBH = 'F48';
xlRange_COC_TOTAL_OUTNM = 'G48';
xlRange_COC_TOTAL_OUTSEAT = 'H48';
xlRange_COC_TOTAL_OUTNMSEAT = 'I48';

xlRange_OPIND = 'C53';
xlRange_IOC_TOT = 'C58';

% Setting actual currency symbol as requasted by the user
if isequal(Cur ,'EUR')
    Cur_Write(1,1) = "[€/Year]";
    Cur_Write(1,2) = "[€/Trip]";
    Cur_Write(1,3) = "[€/FH]";
    Cur_Write(1,4) = "[€/BH]";
    Cur_Write(1,5) = "[€/NM]";
    Cur_Write(1,6) = "[€/Seat]";
    Cur_Write(1,7) = "[€/NM/Seat]";

    Cur_Write_IND = "[€/Year]";
    Cur_Write_TOT = "[€/FH]";

else
    Cur_Write(1,1) = "[US$/Year]";
    Cur_Write(1,2) = "[US$/Trip]";
    Cur_Write(1,3) = "[US$/FH]";
    Cur_Write(1,4) = "[US$/BH]";
    Cur_Write(1,5) = "[US$/NM]";
    Cur_Write(1,6) = "[US$/Seat]";
    Cur_Write(1,7) = "[US$/NM/Seat]";

    Cur_Write_IND = "[US$/Year]";
    Cur_Write_TOT = "[US$/FH]";
end

% Writing OUTPUT Arrays in the 'Results OPERATING' worksheet
xlswrite(file, cellstr(Cur_Write),'Results OPERATING', 'C3');
xlswrite(file, cellstr(Cur_Write_IND),'Results OPERATING', 'C52');
xlswrite(file, cellstr(Cur_Write_TOT),'Results OPERATING', 'L3');

xlswrite(file, OPDIR_OUTYEAR, 'Results OPERATING', xlRange_OPDIR_OUTYEAR)
xlswrite(file, OPDIR_OUTTRIP, 'Results OPERATING', xlRange_OPDIR_OUTTRIP)
xlswrite(file, OPDIR_OUTFH, 'Results OPERATING', xlRange_OPDIR_OUTFH)
xlswrite(file, OPDIR_OUTBH, 'Results OPERATING', xlRange_OPDIR_OUTBH)
xlswrite(file, OPDIR_OUTNM, 'Results OPERATING', xlRange_OPDIR_OUTNM)
xlswrite(file, OPDIR_OUTSEAT, 'Results OPERATING', xlRange_OPDIR_OUTSEAT)
xlswrite(file, OPDIR_OUTNMSEAT, 'Results OPERATING', xlRange_OPDIR_OUTNMSEAT)

xlswrite(file, COST_DOC_TOTAL_OUTYEAR, 'Results OPERATING', xlRange_DOC_TOTAL_OUTYEAR)
xlswrite(file, COST_DOC_TOTAL_OUTTRIP, 'Results OPERATING', xlRange_DOC_TOTAL_OUTTRIP)
xlswrite(file, COST_DOC_TOTAL_OUTFH, 'Results OPERATING', xlRange_DOC_TOTAL_OUTFH)
xlswrite(file, COST_DOC_TOTAL_OUTBH, 'Results OPERATING', xlRange_DOC_TOTAL_OUTBH)
xlswrite(file, COST_DOC_TOTAL_OUTNM, 'Results OPERATING', xlRange_DOC_TOTAL_OUTNM)
xlswrite(file, COST_DOC_TOTAL_OUTSEAT, 'Results OPERATING', xlRange_DOC_TOTAL_OUTSEAT)
xlswrite(file, COST_DOC_TOTAL_OUTNMSEAT, 'Results OPERATING', xlRange_DOC_TOTAL_OUTNMSEAT)

xlswrite(file, COST_COC_TOTAL_OUTYEAR, 'Results OPERATING', xlRange_COC_TOTAL_OUTYEAR)
xlswrite(file, COST_COC_TOTAL_OUTTRIP, 'Results OPERATING', xlRange_COC_TOTAL_OUTTRIP)
xlswrite(file, COST_COC_TOTAL_OUTFH, 'Results OPERATING', xlRange_COC_TOTAL_OUTFH)
xlswrite(file, COST_COC_TOTAL_OUTBH, 'Results OPERATING', xlRange_COC_TOTAL_OUTBH)
xlswrite(file, COST_COC_TOTAL_OUTNM, 'Results OPERATING', xlRange_COC_TOTAL_OUTNM)
xlswrite(file, COST_COC_TOTAL_OUTSEAT, 'Results OPERATING', xlRange_COC_TOTAL_OUTSEAT)
xlswrite(file, COST_COC_TOTAL_OUTNMSEAT, 'Results OPERATING', xlRange_COC_TOTAL_OUTNMSEAT)

xlswrite(file, OPIND, 'Results OPERATING', xlRange_OPIND)
xlswrite(file, COST_IOC_TOTAL,'Results OPERATING', xlRange_IOC_TOT)

winopen(file)

clc
fprintf('-----------------------------------------------------------------------------------\n')
fprintf('Estimation successfull\n')
fprintf('See output file for details\n\n')

toc